/*
 * unmine/board.c
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "board.h"

#include <stdlib.h>

/*
 * Types.
 */

struct Board {
    int width, height;
    Coords last;
    Tile **tiles;
};

/*
 * Private functions.
 */

/*
 * Comparison function for _um_board_reservoir_sample.
 */
static int _um_board_cmp_ints(const void *a, const void *b) {
    return *(const int *) a - *(const int *) b;
}

/*
 * Run the given function on each tile neighbouring the given Tile.
 */
static void _um_board_do_neighbours(const Board *board, const Tile *tile,
                                    void (*func)(const Board *, Tile *)) {
    int i;
    Tile *neighbours[8];
    um_tile_neighbours(board->tiles, tile, board->width, board->height,
                        neighbours);
    for (i = 0; i < 8; ++i) {
        if (neighbours[i]) {
            (*func)(board, neighbours[i]);
        }
    }
}

/*
 * Reveal the given Tile and, if it's a zero Tile, it's neighbours as well.
 */
static void _um_board_expand_tile(const Board *board, Tile *tile) {
    if (!tile->revealed) {
        tile->revealed = 1;
        tile->face = 0;
        if (!tile->count) {
            _um_board_do_neighbours(board, tile, &_um_board_expand_tile);
        }
    }
}

/*
 * Increment the mine count of the given Tile.
 */
static void _um_board_increment_count(const Board *board, Tile *tile) {
    ++tile->count;
}

/*
 * Returns a random integer between 0 and n-1.
 */
static int _um_board_random_int(int n) {
    return (int) ((double) rand() / ((double) RAND_MAX + 1) * n);
}

/*
 * Populate samples with a sorted random sample of the integers between 0 and
 * n-1.
 */
static void _um_board_reservoir_sample(int n, int *samples, int num_samples) {
    int i;
    if (num_samples < 1) {
        return;
    }
    for (i = 0; i < num_samples && i < n; ++i) {
        samples[i] = i;
    }
    if (n <= num_samples) {
        return;
    }
    for (i = num_samples; i < n; ++i) {
        int sample = _um_board_random_int(i + 1);
        if (sample < num_samples) {
            samples[sample] = i;
        }
    }
    qsort(samples, num_samples, sizeof(*samples), _um_board_cmp_ints);
}

/*
 * Set the last property on the Tile at coords.
 */
static void _um_board_set_last(Board *board, const Coords *coords) {
    if (board->last.x != -1 && board->last.y != -1) {
        board->tiles[board->last.x][board->last.y].last = 0;
    }
    board->tiles[coords->x][coords->y].last = 1;
    board->last.x = coords->x;
    board->last.y = coords->y;
}

/*
 * Set the revealed property to the given value, for all Tiles on the Board.
 */
static void _um_board_set_revealed_all(const Board *board, int revealed) {
    int i, j;
    for (i = 0; i < board->width; ++i) {
        for (j = 0; j < board->height; ++j) {
            board->tiles[i][j].revealed = revealed;
            if (revealed) {
                board->tiles[i][j].face = 0;
            }
        }
    }
}

/*
 * Public functions.
 */

/*
 * Create a new Board of the given width and height. Returns a pointer to the
 * new Board, or NULL.
 */
Board * um_board_create(int width, int height) {
    int i, j;
    Board *board;
    Tile **tile_pointers, *tiles;
    board = malloc(sizeof(Board));
    if (!board) {
        return NULL;
    }
    tile_pointers = malloc(sizeof(Tile *) * width);
    if (!tile_pointers) {
        free(board);
        return NULL;
    }
    tiles = malloc(sizeof(Tile) * width * height);
    if (!tiles) {
        free(board);
        free(tile_pointers);
        return NULL;
    }
    for (i = 0; i < width; ++i) {
        tile_pointers[i] = &tiles[i * height];
        for (j = 0; j < height; ++j) {
            tile_pointers[i][j].x = i;
            tile_pointers[i][j].y = j;
        }
    }
    board->width = width;
    board->height = height;
    board->tiles = tile_pointers;
    um_board_clear(board);
    return board;
}

/*
 * Destroy the Board.
 */
void um_board_destroy(Board *board) {
    free(&board->tiles[0][0]);
    free(board->tiles);
    free(board);
}

/*
 * Returns a pointer to the Board's tiles.
 */
Tile ** um_board_tiles(const Board *board) {
    return board->tiles;
}

/*
 * Remove mines, clear faces, reset the last property, and cover all Tiles on
 * the Board.
 */
void um_board_clear(Board *board) {
    int i, j;
    for (i = 0; i < board->width; ++i) {
        for (j = 0; j < board->height; ++j) {
            board->tiles[i][j].revealed = 0;
            board->tiles[i][j].mined = 0;
            board->tiles[i][j].count = 0;
            board->tiles[i][j].face = 0;
            board->tiles[i][j].last = 0;
        }
    }
    board->last.x = -1;
    board->last.y = -1;
}

/*
 * Place a mine at each Tile in coords.
 */
void um_board_mine(const Board *board, const Coords *coords, int num_coords) {
    int i;
    for (i = 0; i < num_coords; ++i) {
        board->tiles[coords[i].x][coords[i].y].mined = 1;
        _um_board_do_neighbours(
            board,
            &board->tiles[coords[i].x][coords[i].y],
            &_um_board_increment_count
        );
    }
}

/*
 * Place mines on the board at random, avoiding the Tiles in coords.
 */
void um_board_mine_random(const Board *board, int num_mines,
                            const Coords *avoids, int num_avoids) {
    int count = 0,
        sample = 0,
        *samples,
        i;
    if (num_mines < 1) {
        return;
    }
    /* Count tiles that can be mined. */
    for (i = 0; i < board->width; ++i) {
        int j;
        for (j = 0; j < board->height; ++j) {
            int avoid = 0,
                k;
            if (board->tiles[i][j].mined) {
                continue;
            }
            for (k = 0; k < num_avoids; ++k) {
                if (i == avoids[k].x && j == avoids[k].y) {
                    avoid = 1;
                    break;
                }
            }
            if (!avoid) {
                ++count;
            }
        }
    }
    /* Avoid adding more mines than the board can hold. */
    if (!count) {
        return;
    }
    if (count < num_mines) {
        num_mines = count;
    }
    /* Take a random sample of tile indexes within the counted range. */
    samples = (int *) malloc(sizeof(int) * num_mines);
    if (!samples) {
        return;
    }
    _um_board_reservoir_sample(count, samples, num_mines);
    /* Iterate through the board, mining the tile at each selected index. */
    count = 0;
    for (i = 0; i < board->width && sample < num_mines; ++i) {
        int j;
        for (j = 0; j < board->height && sample < num_mines; ++j) {
            int avoid = 0,
                k;
            if (board->tiles[i][j].mined) {
                continue;
            }
            for (k = 0; k < num_avoids; ++k) {
                if (i == avoids[k].x && j == avoids[k].y) {
                    avoid = 1;
                    break;
                }
            }
            if (avoid) {
                continue;
            }
            if (count == samples[sample]) {
                board->tiles[i][j].mined = 1;
                _um_board_do_neighbours(board, &board->tiles[i][j],
                                        &_um_board_increment_count);
                ++sample;
            }
            ++count;
            while (sample < num_mines && samples[sample] < count) {
                ++sample;
            }
        }
    }
    free(samples);
}

/*
 * Reveal the Tile at the given coords. Returns the number of surrounding
 * mines, or -1 if the tile is mined. This sets the tile's last property and
 * clears its face.
 */
int um_board_reveal(Board *board, const Coords *coords) {
    Tile *tile = &board->tiles[coords->x][coords->y];
    tile->revealed = 1;
    tile->face = 0;
    _um_board_set_last(board, coords);
    if (tile->mined) {
        return -1;
    }
    return tile->count;
}

/*
 * Cover the Tile at the given coords.
 */
void um_board_cover(const Board *board, const Coords *coords) {
    board->tiles[coords->x][coords->y].revealed = 0;
}

/*
 * Recursively reveal Tiles surrounding zero Tiles, starting with the revealed
 * zero Tile at the given Coords. This also clears Tile faces.
 */
void um_board_expand(const Board *board, const Coords *coords) {
    Tile *tile = &board->tiles[coords->x][coords->y];
    if (!tile->count && tile->revealed) {
        _um_board_do_neighbours(board, tile, &_um_board_expand_tile);
    }
}

/*
 * Set the face of the Tile at coords to the given face_id. This sets the
 * tile's last property.
 */
void um_board_set_face(Board *board, const Coords *coords, int face_id) {
    board->tiles[coords->x][coords->y].face = face_id;
    _um_board_set_last(board, coords);
}

/*
 * Reveal all Tiles on the Board. This also clears Tile faces.
 */
void um_board_reveal_all(const Board *board) {
    _um_board_set_revealed_all(board, 1);
}

/*
 * Cover all Tiles on the Board.
 */
void um_board_cover_all(const Board *board) {
    _um_board_set_revealed_all(board, 0);
}
