/*
 * unmine/board.h
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef UM_BOARD_
#define UM_BOARD_

#include "coords.h"
#include "tile.h"

/*
 * Types.
 */

typedef struct Board Board;

/*
 * Functions.
 */

/*
 * Create a new Board of the given width and height. Returns a pointer to the
 * new Board, or NULL.
 */
Board * um_board_create(int width, int height);

/*
 * Destroy the Board.
 */
void um_board_destroy(Board *board);

/*
 * Returns a pointer to the Board's tiles.
 */
Tile ** um_board_tiles(const Board *board);

/*
 * Remove mines, clear faces, reset the last property, and cover all Tiles on
 * the Board.
 */
void um_board_clear(Board *board);

/*
 * Place a mine at each Tile in coords.
 */
void um_board_mine(const Board *board, const Coords *coords, int num_coords);

/*
 * Place mines on the board at random, avoiding the Tiles in coords.
 */
void um_board_mine_random(const Board *board, int num_mines,
                            const Coords *avoids, int num_avoids);

/*
 * Reveal the Tile at the given coords. Returns the number of surrounding
 * mines, or -1 if the tile is mined. This sets the tile's last property and
 * clears its face.
 */
int um_board_reveal(Board *board, const Coords *coords);

/*
 * Cover the Tile at the given coords.
 */
void um_board_cover(const Board *board, const Coords *coords);

/*
 * Recursively reveal Tiles surrounding zero Tiles, starting with the revealed
 * zero Tile at the given Coords. This also clears Tile faces.
 */
void um_board_expand(const Board *board, const Coords *coords);

/*
 * Set the face of the Tile at coords to the given face_id. This sets the
 * tile's last property.
 */
void um_board_set_face(Board *board, const Coords *coords, int face_id);

/*
 * Reveal all Tiles on the Board. This also clears Tile faces.
 */
void um_board_reveal_all(const Board *board);

/*
 * Cover all Tiles on the Board.
 */
void um_board_cover_all(const Board *board);

#endif /* UM_BOARD_ */
