/*
 * unmine/classic.c
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "classic.h"

#include <stdlib.h>
#include <sys/types.h>
#include <time.h>

#include "board.h"

/*
 * Constants.
 */

const ClassicParameters UM_CLASSIC_MIN     = {2,    2,    1};
const ClassicParameters UM_CLASSIC_DEFAULT = {8,    8,    10};
const ClassicParameters UM_CLASSIC_SMALL   = {8,    8,    10};
const ClassicParameters UM_CLASSIC_MEDIUM  = {16,   16,   40};
const ClassicParameters UM_CLASSIC_LARGE   = {30,   16,   99};
const ClassicParameters UM_CLASSIC_MAX     = {4096, 4096, 16777215};

/*
 * Types.
 */

enum CheckHint {
    NO_HINT, CHANGE_FLAGS, CHANGE_REVEAL, SKIP_FLAGS, SKIP_REVEAL
};

struct ClassicGame {
    ClassicCallbacks callbacks;
    Board *board;
    Tile **tiles;
    ClassicOptions options;
    ClassicParameters params;
    ClassicState state;
    time_t start_time,
           stop_time;
    int flag_count;
};

/*
 * Private functions.
 */

/*
 * Returns 1 if all mines are flagged and all non-mined tiles are not flagged.
 */
static int _um_classic_is_win_flags(int width, int height, Tile **tiles) {
    int i, j;
    for (i = 0; i < width; ++i) {
        for (j = 0; j < height; ++j) {
            if (tiles[i][j].mined != (tiles[i][j].face == MINE_FLAG)) {
                return 0;
            }
        }
    }
    return 1;
}

/*
 * Returns 1 if all non-mined tiles are revealed.
 */
static int _um_classic_is_win_reveal(int width, int height, Tile **tiles) {
    int i, j;
    for (i = 0; i < width; ++i) {
        for (j = 0; j < height; ++j) {
            if (!tiles[i][j].mined && !tiles[i][j].revealed) {
                return 0;
            }
        }
    }
    return 1;
}

/*
 * Call the user's update_board callback, if provided.
 */
static void _um_classic_update_board(ClassicGame *game) {
    if (game->callbacks.update_board) {
        game->callbacks.update_board(game->tiles, game->callbacks.data);
    }
}

/*
 * Call the user's update_flags callback, if provided.
 */
static void _um_classic_update_flags(ClassicGame *game) {
    if (game->callbacks.update_flags) {
        game->callbacks.update_flags(game->flag_count, game->callbacks.data);
    }
}

/*
 * Call the user's update_state callback, if provided.
 */
static void _um_classic_update_state(ClassicGame *game) {
    if (game->callbacks.update_state) {
        game->callbacks.update_state(game->state, game->callbacks.data);
    }
}

/*
 * Call the user's update_tile callback with the given tile, if provided.
 */
static void _um_classic_update_tile(ClassicGame *game, Tile *tile) {
    if (game->callbacks.update_tile) {
        game->callbacks.update_tile(tile, game->callbacks.data);
    }
}

/*
 * Check if the game has been won. Optionally, to skip over a check that isn't
 * needed, specify a hint:
 *
 *      CHANGE_FLAGS and CHANGE_REVEAL indicate the only change to the game
 *      since the last check.
 *
 *      SKIP_FLAGS and SKIP_REVEAL indicate that a flag or reveal check *will*
 *      fail.
 */
static void _um_classic_check_win(ClassicGame *game, int check_hint) {
    int mode = game->options.win_mode,
        win = 0;
    if (mode == REVEAL) {
        if (check_hint != CHANGE_FLAGS && check_hint != SKIP_REVEAL) {
            win = _um_classic_is_win_reveal(game->params.width,
                                            game->params.height, game->tiles);
        }
    } else if (mode == FLAGS) {
        if (check_hint != CHANGE_REVEAL && check_hint != SKIP_FLAGS) {
            win = _um_classic_is_win_flags(game->params.width,
                                            game->params.height, game->tiles);
        }
    } else if (mode == EITHER) {
        if (check_hint != CHANGE_FLAGS && check_hint != SKIP_REVEAL) {
            win = _um_classic_is_win_reveal(game->params.width,
                                            game->params.height, game->tiles);
        }
        if (!win && check_hint != CHANGE_REVEAL && check_hint != SKIP_FLAGS) {
            win = _um_classic_is_win_flags(game->params.width,
                                            game->params.height, game->tiles);
        }
    } else if (mode == BOTH) {
        if (check_hint != SKIP_FLAGS && check_hint != SKIP_REVEAL) {
            win = _um_classic_is_win_reveal(
                game->params.width,
                game->params.height,
                game->tiles
            ) && _um_classic_is_win_flags(
                game->params.width,
                game->params.height,
                game->tiles
            );
        }
    }
    if (win) {
        game->state = WON;
        game->stop_time = time(NULL);
        _um_classic_update_state(game);
    }
}

/*
 * Set the face of incorrectly flagged tiles to INCORRECT_FLAG.
 */
static void _um_classic_mark_incorrect(const ClassicGame *game, int hide) {
    int i, j;
    for (i = 0; i < game->params.width; ++i) {
        for (j = 0; j < game->params.height; ++j) {
            if (!game->tiles[i][j].mined) {
                if (hide) {
                    if (game->tiles[i][j].face == INCORRECT_FLAG) {
                        game->tiles[i][j].face = MINE_FLAG;
                    }
                } else {
                    if (game->tiles[i][j].face == MINE_FLAG) {
                        game->tiles[i][j].face = INCORRECT_FLAG;
                    }
                }
            }
        }
    }
}

/*
 * Switch to a new board and mine it (this does not destroy the old board).
 * This resets the game (but does not call um_classic_reset).
 */
static void _um_classic_new_board(ClassicGame *game, Board *new_board) {
    game->board = new_board;
    game->tiles = um_board_tiles(new_board);
    game->flag_count = game->params.num_mines;
    game->state = NOT_STARTED;
    um_board_mine_random(new_board, game->params.num_mines, NULL, 0);
}

/*
 * Reveal (or cover) unflagged mines (except the last Tile, which is always
 * shown).
 */
static void _um_classic_reveal_mined(const ClassicGame *game, int cover) {
    int i, j,
        new_val = 1;
    if (cover) {
        new_val = 0;
    }
    for (i = 0; i < game->params.width; i += 1) {
        for (j = 0; j < game->params.height; j += 1) {
            if (game->tiles[i][j].mined &&
                game->tiles[i][j].face != MINE_FLAG &&
                !game->tiles[i][j].last) {
                game->tiles[i][j].revealed = new_val;
            }
        }
    }
}

/*
 * Declare the game LOST.
 */
static void _um_classic_lost(ClassicGame *game) {
    game->state = LOST;
    game->stop_time = time(NULL);
    if (!game->options.cover_on_loss) {
        _um_classic_mark_incorrect(game, 0);
        _um_classic_reveal_mined(game, 0);
    }
    _um_classic_update_state(game);
}

/*
 * Toggle the given face on the tile at the given coords. Returns 1 if the tile
 * now has the given face.
 */
static int _um_classic_toggle_face(ClassicGame *game, const Coords *coords,
                                    ClassicFace face) {
    ClassicFace current = game->tiles[coords->x][coords->y].face;
    int result = 1;
    if (current == face) {
        face = BLANK;
        result = 0;
    }
    um_board_set_face(game->board, coords, face);
    if (current == MINE_FLAG) {
        ++game->flag_count;
    } else if (face == MINE_FLAG) {
        --game->flag_count;
    }
    if (current == MINE_FLAG || face == MINE_FLAG) {
        _um_classic_check_win(game, CHANGE_FLAGS);
        _um_classic_update_flags(game);
    }
    _um_classic_update_tile(game, &game->tiles[coords->x][coords->y]);
    return result;
}

/*
 * Transition the game between on and off states for the cover_on_loss option.
 */
static void _um_classic_transition_cover_on_loss(ClassicGame *game,
                                                    int old, int new) {
    if (game->state == LOST) {
        if (old && !(new)) {
            _um_classic_mark_incorrect(game, 0);
            _um_classic_reveal_mined(game, 0);
        } else if (!(old) && new) {
            _um_classic_mark_incorrect(game, 1);
            _um_classic_reveal_mined(game, 1);
        }
        _um_classic_update_board(game);
    }
}

/*
 * Transition the game between on and off states for the no_expand_zero_tiles
 * option.
 */
static void _um_classic_transition_no_expand_zero_tiles(ClassicGame *game,
                                                        int old, int new) {
    if (old && !(new)) {
        int change = 0,
            i, j;
        for (i = 0; i < game->params.width; ++i) {
            for (j = 0; j < game->params.height; ++j) {
                if (!game->tiles[i][j].count && game->tiles[i][j].revealed &&
                    !game->tiles[i][j].mined) {
                    Coords coords;
                    coords.x = i;
                    coords.y = j;
                    um_board_expand(game->board, &coords);
                    change = 1;
                }
            }
        }
        if (change) {
            if (game->state == IN_PROGRESS) {
                _um_classic_check_win(game, CHANGE_REVEAL);
            }
            _um_classic_update_board(game);
        }
    } else if (!(old) && new) {
        /*
         * No action required; it's not possible to un-expand zero tiles.
         */
    }
}

/*
 * Transition the game between on and off states for the question_marks
 * option.
 */
static void _um_classic_transition_question_marks(ClassicGame *game,
                                                        int old, int new) {
    if (old && !(new)) {
        int i, j;
        for (i = 0; i < game->params.width; ++i) {
            for (j = 0; j < game->params.height; ++j) {
                if (!game->tiles[i][j].revealed &&
                    game->tiles[i][j].face == QUESTION_MARK) {
                    game->tiles[i][j].face = BLANK;
                    _um_classic_update_tile(game, &game->tiles[i][j]);
                }
            }
        }
    } else if (!(old) && new) {
        /*
         * No action required; nothing happens when enabling question marks.
         */
    }
}

/*
 * Transition the Game between old and new WinModes.
 */
static void _um_classic_transition_win_mode(ClassicGame *game,
                                            ClassicWinMode old,
                                            ClassicWinMode new) {
    if (game->state == IN_PROGRESS && new != BOTH) {
        /*
         * It's not possible for the game to be considered WON upon changing to
         * BOTH.
         */
        if (old == REVEAL) {
            /*
             * One or more non-mined tiles are covered. Regardless of the new
             * win mode, the game can only now be considered won if all the
             * mines are flagged.
             */
            _um_classic_check_win(game, SKIP_REVEAL);
        } else if (old == FLAGS) {
            /*
             * One or more mines are not flagged. Regardless of the new
             * win mode, the game can only now be considered won if all the
             * non-mined tiles are revealed.
             */
            _um_classic_check_win(game, SKIP_FLAGS);
        } else if (old == EITHER) {
            /*
             * No action required. One or more non-mined tiles are covered, and
             * one or more mines are not flagged, therefore it's not possible
             * for the game to now be considered won.
             */
        } else if (old == BOTH) {
            /*
             * Either one or more non-mined tiles are covered, or one or more
             * mines are not flagged. Depending on the new win mode, the game
             * may be won in multiple ways.
             */
            _um_classic_check_win(game, NO_HINT);
        }
    }
}

/*
 * Public functions.
 */

/*
 * Create a new game with the given callbacks, options, and parameters. Returns
 * a pointer to the new game, or NULL.
 */
ClassicGame * um_classic_create(const ClassicCallbacks *callbacks,
                                const ClassicOptions *options,
                                const ClassicParameters *params) {
    Board *board;
    ClassicGame *game;
    board = um_board_create(params->width, params->height);
    if (!board) {
        return NULL;
    }
    game = malloc(sizeof(ClassicGame));
    if (!game) {
        um_board_destroy(board);
        return NULL;
    }
    game->callbacks = *callbacks;
    game->options = *options;
    game->params = *params;
    _um_classic_new_board(game, board);
    return game;
}

/*
 * Destroy the game.
 */
void um_classic_destroy(ClassicGame *game) {
    um_board_destroy(game->board);
    free(game);
}

/*
 * Calls your update_board, update_flags, and update_state callback functions,
 * allowing your interface to refresh when the game hasn't changed. You should
 * rarely need to use this.
 */
void um_classic_update(ClassicGame *game) {
    _um_classic_update_board(game);
    _um_classic_update_flags(game);
    _um_classic_update_state(game);
}

/*
 * Returns the number of mine flags remaining.
 */
int um_classic_flag_count(const ClassicGame *game) {
    return game->flag_count;
}

/*
 * Returns the current game state.
 */
ClassicState um_classic_state(const ClassicGame *game) {
    return game->state;
}

/*
 * Returns a pointer to the game's tiles.
 */
Tile ** um_classic_tiles(const ClassicGame *game) {
    return game->tiles;
}

/*
 * Returns the game time. If the game is NOT_STARTED, this returns 0. If the
 * game is IN_PROGRESS, this returns the number of seconds between the time the
 * game was started and now. If the game is LOST or WON, this returns the
 * number of seconds between the time the game was started and the time the
 * game was LOST or WON.
 */
double um_classic_time(const ClassicGame *game) {
    if (game->state == NOT_STARTED) {
        return 0;
    } else if (game->state == IN_PROGRESS) {
        return difftime(time(NULL), game->start_time);
    } else {
        return difftime(game->stop_time, game->start_time);
    }
}

/*
 * Set the game's options. Changing the no_expand_zero_tiles option may reveal
 * one or more tiles or declare the game WON (in all win modes except FLAGS),
 * changing the cover_on_loss option will reveal or cover tiles if the game is
 * LOST, disabling the question_marks option will remove all question marks on
 * the board, and changing the win mode may declare the game WON. This does not
 * reset the game.
 */
void um_classic_options(ClassicGame *game, const ClassicOptions *new_opts) {
    ClassicOptions old_opts = game->options;
    game->options = *new_opts;
    if (old_opts.no_expand_zero_tiles != new_opts->no_expand_zero_tiles) {
        _um_classic_transition_no_expand_zero_tiles(
            game,
            old_opts.no_expand_zero_tiles,
            new_opts->no_expand_zero_tiles
        );
    }
    if (old_opts.cover_on_loss != new_opts->cover_on_loss) {
        _um_classic_transition_cover_on_loss(
            game,
            old_opts.cover_on_loss,
            new_opts->cover_on_loss
        );
    }
    if (old_opts.question_marks != new_opts->question_marks) {
        _um_classic_transition_question_marks(
            game,
            old_opts.question_marks,
            new_opts->question_marks
        );
    }
    if (old_opts.win_mode != new_opts->win_mode) {
        _um_classic_transition_win_mode(
            game,
            old_opts.win_mode,
            new_opts->win_mode
        );
    }
}

/*
 * Set the size of the game's board and/or the number of mines, resetting the
 * game. Returns a pointer to the new tiles (this voids any previous pointers
 * returned by this function, or um_classic_tiles), or NULL (in which case, the
 * game is destroyed).
 */
Tile ** um_classic_parameters(ClassicGame *game,
                                const ClassicParameters *params) {
    if (params->width != game->params.width ||
        params->height != game->params.height) {
        Board *new_board;
        um_board_destroy(game->board);
        new_board = um_board_create(params->width, params->height);
        if (!new_board) {
            free(game);
            return NULL;
        }
        game->params = *params;
        _um_classic_new_board(game, new_board);
    } else if (params->num_mines != game->params.num_mines) {
        game->params.num_mines = params->num_mines;
        um_classic_reset(game);
    }
    return game->tiles;
}

/*
 * Reset the game.
 */
void um_classic_reset(ClassicGame *game) {
    um_board_clear(game->board);
    game->flag_count = game->params.num_mines;
    game->state = NOT_STARTED;
    um_board_mine_random(game->board, game->params.num_mines, NULL, 0);
    um_classic_update(game);
}

/*
 * Reveal the tile at the given coords. Returns -2 if the game is already LOST
 * or WON or the tile is already revealed. Returns -1 if the tile is mined.
 * Otherwise, returns the number of surrounding mines. This may reveal one or
 * more other tiles if the no_expand_zero_tiles option is off (in which case,
 * this will return 0), or declare the game WON (in all win modes except
 * FLAGS), or LOST (in which case, this will return -1).
 */
int um_classic_reveal(ClassicGame *game, const Coords *coords) {
    int reveal_result,
        update_board = 0;
    if (game->state == LOST || game->state == WON ||
        game->tiles[coords->x][coords->y].revealed) {
        return -2;
    }
    if (game->state == NOT_STARTED) {
        game->state = IN_PROGRESS;
        game->start_time = time(NULL);
        _um_classic_update_state(game);
    }
    reveal_result = um_board_reveal(game->board, coords);
    if (reveal_result == -1) {
        _um_classic_lost(game);
        update_board = 1;
    } else {
        if (!reveal_result && !game->options.no_expand_zero_tiles) {
            um_board_expand(game->board, coords);
            update_board = 1;
        }
        _um_classic_check_win(game, CHANGE_REVEAL);
    }
    if (update_board) {
        _um_classic_update_board(game);
    } else {
        _um_classic_update_tile(game, &game->tiles[coords->x][coords->y]);
    }
    return reveal_result;
}

/*
 * Reveal tiles surrounding the tile at the given coords, when the number of
 * flagged tiles surrounding the tile is equal to the tile's mine count.
 * This may reveal one or more other tiles if the no_expand_zero_tiles option
 * is off, or declare the game WON (in all win modes except FLAGS) or LOST.
 */
void um_classic_reveal_around(ClassicGame *game, const Coords *coords) {
    int i,
        lost = 0,
        mine_count = game->tiles[coords->x][coords->y].count,
        update_board = 0;
    Tile *neighbours[8];
    if (game->state != IN_PROGRESS ||
        !game->tiles[coords->x][coords->y].revealed) {
        return;
    }
    um_tile_neighbours(game->tiles, &game->tiles[coords->x][coords->y],
                        game->params.width, game->params.height, neighbours);
    if (mine_count) {
        int flag_count = 0;
        for (i = 0; i < 8; ++i) {
            if (neighbours[i] && neighbours[i]->face == MINE_FLAG) {
                ++flag_count;
            }
        }
        if (flag_count != mine_count) {
            return;
        }
    }
    for (i = 0; i < 8; ++i) {
        if (neighbours[i] && neighbours[i]->face != MINE_FLAG) {
            Coords ncoords;
            int reveal_result;
            ncoords.x = neighbours[i]->x;
            ncoords.y = neighbours[i]->y;
            reveal_result = um_board_reveal(game->board, &ncoords);
            if (reveal_result == -1) {
                lost = 1;
            } else if (!reveal_result && !game->options.no_expand_zero_tiles) {
                um_board_expand(game->board, &ncoords);
                update_board = 1;
            }
        }
    }
    if (lost) {
        _um_classic_lost(game);
    } else {
        _um_classic_check_win(game, CHANGE_REVEAL);
    }
    for (i = 0; i < 8; ++i) {
        if (neighbours[i] && neighbours[i]->face != MINE_FLAG) {
            neighbours[i]->last = neighbours[i]->mined;
            if (!lost && !update_board) {
                _um_classic_update_tile(game, neighbours[i]);
            }
        }
    }
    if (lost || update_board) {
        _um_classic_update_board(game);
    }
}

/*
 * Toggle the flag on the given coords. Returns 1 if the tile now has a flag.
 * This may declare the game WON (in all win modes except REVEAL).
 */
int um_classic_flag(ClassicGame *game, const Coords *coords) {
    if (game->state == LOST || game->state == WON ||
        game->tiles[coords->x][coords->y].revealed) {
        return 0;
    }
    return _um_classic_toggle_face(game, coords, MINE_FLAG);
}

/*
 * Toggle the question mark on the given coords. The question_marks option must
 * be enabled. Returns 1 if the tile now has a question mark.
 */
int um_classic_question(ClassicGame *game, const Coords *coords) {
    if (game->state == LOST || game->state == WON ||
        game->tiles[coords->x][coords->y].revealed ||
        !game->options.question_marks) {
        return 0;
    }
    return _um_classic_toggle_face(game, coords, QUESTION_MARK);
}
