/*
 * unmine/classic.h
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef UM_CLASSIC_
#define UM_CLASSIC_

#include "coords.h"
#include "tile.h"

/*
 * Types.
 */

typedef struct ClassicGame ClassicGame;

typedef enum {
    BLANK, MINE_FLAG, QUESTION_MARK, INCORRECT_FLAG
} ClassicFace;

typedef enum {
    NOT_STARTED, IN_PROGRESS, LOST, WON
} ClassicState;

typedef enum {
    REVEAL, FLAGS, EITHER, BOTH
} ClassicWinMode;

typedef struct {
    void (*update_board)(Tile **tiles, void *data);
    void (*update_flags)(int flags, void *data);
    void (*update_state)(ClassicState state, void *data);
    void (*update_tile)(Tile *tile, void *data);
    void *data;
} ClassicCallbacks;

typedef struct {
    int cover_on_loss, no_expand_zero_tiles, question_marks;
    ClassicWinMode win_mode;
} ClassicOptions;

typedef struct {
    int width, height, num_mines;
} ClassicParameters;

/*
 * Constants.
 */

extern const ClassicParameters UM_CLASSIC_MIN;
extern const ClassicParameters UM_CLASSIC_DEFAULT;
extern const ClassicParameters UM_CLASSIC_SMALL;
extern const ClassicParameters UM_CLASSIC_MEDIUM;
extern const ClassicParameters UM_CLASSIC_LARGE;
extern const ClassicParameters UM_CLASSIC_MAX;

/*
 * Functions.
 */

/*
 * Create a new game with the given callbacks, options, and parameters. Returns
 * a pointer to the new game, or NULL.
 */
ClassicGame * um_classic_create(const ClassicCallbacks *callbacks,
                                const ClassicOptions *options,
                                const ClassicParameters *params);

/*
 * Destroy the game.
 */
void um_classic_destroy(ClassicGame *game);

/*
 * Calls your update_board, update_flags, and update_state callback functions,
 * allowing your interface to refresh when the game hasn't changed. You should
 * rarely need to use this.
 */
void um_classic_update(ClassicGame *game);

/*
 * Returns the number of mine flags remaining.
 */
int um_classic_flag_count(const ClassicGame *game);

/*
 * Returns the current game state.
 */
ClassicState um_classic_state(const ClassicGame *game);

/*
 * Returns a pointer to the game's tiles.
 */
Tile ** um_classic_tiles(const ClassicGame *game);

/*
 * Returns the game time. If the game is NOT_STARTED, this returns 0. If the
 * game is IN_PROGRESS, this returns the number of seconds between the time the
 * game was started and now. If the game is LOST or WON, this returns the
 * number of seconds between the time the game was started and the time the
 * game was LOST or WON.
 */
double um_classic_time(const ClassicGame *game);

/*
 * Set the game's options. Changing the no_expand_zero_tiles option may reveal
 * one or more tiles or declare the game WON (in all win modes except FLAGS),
 * changing the cover_on_loss option will reveal or cover tiles if the game is
 * LOST, disabling the question_marks option will remove all question marks on
 * the board, and changing the win mode may declare the game WON. This does not
 * reset the game.
 */
void um_classic_options(ClassicGame *game, const ClassicOptions *options);

/*
 * Set the size of the game's board and/or the number of mines, resetting the
 * game. Returns a pointer to the new tiles (this voids any previous pointers
 * returned by this function, or um_classic_tiles), or NULL (in which case, the
 * game is destroyed).
 */
Tile ** um_classic_parameters(ClassicGame *game,
                                const ClassicParameters *params);

/*
 * Reset the game.
 */
void um_classic_reset(ClassicGame *game);

/*
 * Reveal the tile at the given coords. Returns -2 if the game is already LOST
 * or WON or the tile is already revealed. Returns -1 if the tile is mined.
 * Otherwise, returns the number of surrounding mines. This may reveal one or
 * more other tiles if the no_expand_zero_tiles option is off (in which case,
 * this will return 0), or declare the game WON (in all win modes except
 * FLAGS), or LOST (in which case, this will return -1).
 */
int um_classic_reveal(ClassicGame *game, const Coords *coords);

/*
 * Reveal tiles surrounding the tile at the given coords, when the number of
 * flagged tiles surrounding the tile is equal to the tile's mine count.
 * This may reveal one or more other tiles if the no_expand_zero_tiles option
 * is off, or declare the game WON (in all win modes except FLAGS) or LOST.
 */
void um_classic_reveal_around(ClassicGame *game, const Coords *coords);

/*
 * Toggle the flag on the given coords. Returns 1 if the tile now has a flag.
 * This may declare the game WON (in all win modes except REVEAL).
 */
int um_classic_flag(ClassicGame *game, const Coords *coords);

/*
 * Toggle the question mark on the given coords. The question_marks option must
 * be enabled. Returns 1 if the tile now has a question mark.
 */
int um_classic_question(ClassicGame *game, const Coords *coords);

#endif /* UM_CLASSIC_ */
