/*
 * unmine/classic_config.c
 *
 * Copyright 2017 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#define _POSIX_C_SOURCE 199506L

#include "classic_config.h"

#include <strings.h>

/*
 * Strings.
 */

static const char *NAME_COVER_ON_LOSS = "cover_on_loss";
static const char *NAME_NO_EXPAND_ZERO_TILES = "no_expand_zero_tiles";
static const char *NAME_QUESTION_MARKS = "question_marks";
static const char *NAME_WIN_MODE = "win_mode";

static const char *NAME_WIDTH = "width";
static const char *NAME_HEIGHT = "height";
static const char *NAME_NUM_MINES = "num_mines";

static const char * const VALUES_WIN_MODE[] = {
    "reveal", "flags", "either", "both"
};

/*
 * Public functions.
 */

/*
 * Add configuration settings for classic game options to the given group. This
 * can be called repeatedly with the same group to populate missing settings.
 */
void um_classic_config_create_options(config_setting_t *group) {
    if (!config_setting_get_member(group, NAME_COVER_ON_LOSS)) {
        config_setting_add(group, NAME_COVER_ON_LOSS, CONFIG_TYPE_BOOL);
    }
    if (!config_setting_get_member(group, NAME_NO_EXPAND_ZERO_TILES)) {
        config_setting_add(group, NAME_NO_EXPAND_ZERO_TILES, CONFIG_TYPE_BOOL);
    }
    if (!config_setting_get_member(group, NAME_QUESTION_MARKS)) {
        config_setting_add(group, NAME_QUESTION_MARKS, CONFIG_TYPE_BOOL);
    }
    if (!config_setting_get_member(group, NAME_WIN_MODE)) {
        config_setting_add(group, NAME_WIN_MODE, CONFIG_TYPE_STRING);
    }
}

/*
 * Add configuration settings for classic game parameters to the given group.
 * This can be called repeatedly with the same group to populate missing
 * settings.
 */
void um_classic_config_create_parameters(config_setting_t *group) {
    if (!config_setting_get_member(group, NAME_WIDTH)) {
        config_setting_add(group, NAME_WIDTH, CONFIG_TYPE_INT);
    }
    if (!config_setting_get_member(group, NAME_HEIGHT)) {
        config_setting_add(group, NAME_HEIGHT, CONFIG_TYPE_INT);
    }
    if (!config_setting_get_member(group, NAME_NUM_MINES)) {
        config_setting_add(group, NAME_NUM_MINES, CONFIG_TYPE_INT);
    }
}

/*
 * Convenience function; calls functions um_classic_config_load_*, and calls
 * functions um_classic_config_create_* to fill in missing settings.
 */
void um_classic_config_load(config_setting_t *group, ClassicOptions *options,
                            ClassicParameters *params) {
    if (um_classic_config_load_options(group, options)) {
        um_classic_config_create_options(group);
    }
    if (um_classic_config_load_parameters(group, params)) {
        um_classic_config_create_parameters(group);
    }
}

/*
 * Load classic game options from the configuration group. No changes are made
 * to options with missing or invalid settings. Returns 1 if one or more
 * settings are missing or invalid, otherwise 0.
 */
int um_classic_config_load_options(config_setting_t *group,
                                    ClassicOptions *options) {
    int result_col,
        result_nezt,
        result_qm;
    const char *wm;
    result_col = config_setting_lookup_bool(group, NAME_COVER_ON_LOSS,
                                            &options->cover_on_loss);
    result_nezt = config_setting_lookup_bool(group, NAME_NO_EXPAND_ZERO_TILES,
                                            &options->no_expand_zero_tiles);
    result_qm = config_setting_lookup_bool(group, NAME_QUESTION_MARKS,
                                            &options->question_marks);
    if (config_setting_lookup_string(group, NAME_WIN_MODE, &wm) ==
                                                                CONFIG_FALSE) {
        return 1;
    }
    if (!strcasecmp(wm, VALUES_WIN_MODE[REVEAL])) {
        options->win_mode = REVEAL;
    } else if (!strcasecmp(wm, VALUES_WIN_MODE[FLAGS])) {
        options->win_mode = FLAGS;
    } else if (!strcasecmp(wm, VALUES_WIN_MODE[EITHER])) {
        options->win_mode = EITHER;
    } else if (!strcasecmp(wm, VALUES_WIN_MODE[BOTH])) {
        options->win_mode = BOTH;
    }
    return result_col == CONFIG_FALSE ||
           result_nezt == CONFIG_FALSE ||
           result_qm == CONFIG_FALSE;
}

/*
 * Load classic game parameters from the configuration group. No changes are
 * made to parameters with missing or invalid settings. Returns 1 if one or
 * more settings are missing or invalid, otherwise 0.
 */
int um_classic_config_load_parameters(config_setting_t *group,
                                        ClassicParameters *params) {
    int result_w,
        result_h,
        result_n,
        width = 0,
        height = 0,
        num_mines = 0;
    result_w = config_setting_lookup_int(group, NAME_WIDTH, &width);
    if (result_w == CONFIG_TRUE &&
        width >= UM_CLASSIC_MIN.width &&
        width <= UM_CLASSIC_MAX.width) {
        params->width = width;
    }
    result_h = config_setting_lookup_int(group, NAME_HEIGHT, &height);
    if (result_h == CONFIG_TRUE &&
        height >= UM_CLASSIC_MIN.height &&
        height <= UM_CLASSIC_MAX.height) {
        params->height = height;
    }
    result_n = config_setting_lookup_int(group, NAME_NUM_MINES, &num_mines);
    if (result_n == CONFIG_TRUE &&
        num_mines >= UM_CLASSIC_MIN.num_mines &&
        num_mines <= UM_CLASSIC_MAX.num_mines &&
        num_mines < (width * height - 2)) {
        params->num_mines = num_mines;
    }
    return result_w == CONFIG_FALSE ||
           result_h == CONFIG_FALSE ||
           result_n == CONFIG_FALSE;
}

/*
 * Convenience function; calls functions um_classic_config_save_*.
 */
void um_classic_config_save(config_setting_t *group, ClassicOptions *options,
                            ClassicParameters *params) {
    um_classic_config_save_options(group, options);
    um_classic_config_save_parameters(group, params);
}

/*
 * Save classic game options into the configuration group. No changes are made
 * to missing or invalid settings. Returns 1 if one or more settings are
 * missing or invalid, otherwise 0.
 */
int um_classic_config_save_options(config_setting_t *group,
                                    ClassicOptions *options) {
    config_setting_t *setting_col,
                     *setting_nezt,
                     *setting_qm,
                     *setting_wm;
    int result_col = CONFIG_FALSE,
        result_nezt = CONFIG_FALSE,
        result_qm = CONFIG_FALSE,
        result_wm = CONFIG_FALSE;
    setting_col = config_setting_get_member(group, NAME_COVER_ON_LOSS);
    if (setting_col) {
        result_col = config_setting_set_bool(setting_col,
                                                options->cover_on_loss);
    }
    setting_nezt = config_setting_get_member(group, NAME_NO_EXPAND_ZERO_TILES);
    if (setting_nezt) {
        result_nezt = config_setting_set_bool(setting_nezt,
                                                options->no_expand_zero_tiles);
    }
    setting_qm = config_setting_get_member(group, NAME_QUESTION_MARKS);
    if (setting_qm) {
        result_qm = config_setting_set_bool(setting_qm,
                                            options->question_marks);
    }
    setting_wm = config_setting_get_member(group, NAME_WIN_MODE);
    if (setting_wm) {
        result_wm = config_setting_set_string(setting_wm,
                                        VALUES_WIN_MODE[options->win_mode]);
    }
    return result_col  == CONFIG_FALSE ||
           result_nezt == CONFIG_FALSE ||
           result_qm   == CONFIG_FALSE ||
           result_wm   == CONFIG_FALSE;
}

/*
 * Save classic game parameters into the configuration group. No changes are
 * made to missing or invalid settings. Returns 1 if one or more settings are
 * missing or invalid, otherwise 0.
 */
int um_classic_config_save_parameters(config_setting_t *group,
                                        ClassicParameters *params) {
    config_setting_t *setting_w,
                     *setting_h,
                     *setting_n;
    int result_w = CONFIG_FALSE,
        result_h = CONFIG_FALSE,
        result_n = CONFIG_FALSE;
    setting_w = config_setting_get_member(group, NAME_WIDTH);
    if (setting_w) {
        result_w = config_setting_set_int(setting_w, params->width);
    }
    setting_h = config_setting_get_member(group, NAME_HEIGHT);
    if (setting_h) {
        result_h = config_setting_set_int(setting_h, params->height);
    }
    setting_n = config_setting_get_member(group, NAME_NUM_MINES);
    if (setting_n) {
        result_n = config_setting_set_int(setting_n, params->num_mines);
    }
    return result_w == CONFIG_FALSE ||
           result_h == CONFIG_FALSE ||
           result_n == CONFIG_FALSE;
}
