/*
 * unmine/flags.c
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#define _POSIX_C_SOURCE 199506L

#include "flags.h"

#include <stdlib.h>

#include <pthread.h>
#include <unistd.h>

#include "board.h"

/*
 * Constants.
 */

const FlagsParameters UM_FLAGS_MIN     = {2,    2,    1,        1, 0, NULL};
const FlagsParameters UM_FLAGS_DEFAULT = {16,   16,   51,       2, 0, NULL};
const FlagsParameters UM_FLAGS_MAX     = {4096, 4096, 16777215, 8, 8, NULL};

/*
 * Types.
 */

typedef enum {
    DEAD, ALIVE
} AIThreadState;

typedef enum {
    DIE, WAIT, PLAY
} AIThreadTask;

struct FlagsGame {
    FlagsCallbacks callbacks;
    FlagsParameters params;
    FlagsOptions options;
    Board *board;
    Tile **tiles;
    FlagsPlayer *players;
    FlagsState state;
    int mine_count,
        turn_player;
    pthread_cond_t ai_task_ready;
    pthread_mutex_t global_mutex;
    AIThreadState ai_state;
    AIThreadTask ai_task;
};

/*
 * Private functions.
 */

static void   _um_flags_ai_task_cancel(FlagsGame *game);
static void   _um_flags_ai_task_set(FlagsGame *game);
static int    _um_flags_ai_thread_check(FlagsGame *game);
static void * _um_flags_ai_thread_run(void *data);
static int    _um_flags_ai_thread_start(FlagsGame *game);
static void   _um_flags_ai_thread_stop(FlagsGame *game, int wait);
static int    _um_flags_bomb(FlagsGame *game, int player,
                                const Coords *coords);
static int    _um_flags_check_ai_levels(FlagsGame *game);
static int    _um_flags_check_win(FlagsGame *game);
static int    _um_flags_new_board(FlagsGame *game);
static int    _um_flags_new_players(FlagsGame *game);
static void   _um_flags_next_turn(FlagsGame *game);
static void   _um_flags_reset_all(FlagsGame *game, int skip_board,
                                    int skip_players);
static void   _um_flags_reset_ai_levels(FlagsGame *game);
static void   _um_flags_reset_board(FlagsGame *game);
static void   _um_flags_reset_players(FlagsGame *game);
static int    _um_flags_reveal(FlagsGame *game, int player,
                                const Coords *coords);
static int    _um_flags_reveal_end(FlagsGame *game, const Coords *coords,
                                int update_board, int update_players, int hit,
                                int bomb);
static void   _um_flags_reveal_mined(FlagsGame *game, int cover);
static int    _um_flags_select(FlagsGame *game, int player,
                                const Coords *coords);
static void   _um_flags_transition_allow_ties(FlagsGame *game, int old,
                                                int new);
static void   _um_flags_transition_cover_on_done(FlagsGame *game, int old,
                                                    int new);
static void   _um_flags_transition_no_expand_zero_tiles(FlagsGame *game,
                                                        int old, int new);
static void   _um_flags_transition_num_bombs(FlagsGame *game, int new);
static void   _um_flags_update_board(FlagsGame *game);
static void   _um_flags_update_mine_count(FlagsGame *game);
static void   _um_flags_update_players(FlagsGame *game);
static void   _um_flags_update_state(FlagsGame *game);
static void   _um_flags_update_tile(FlagsGame *game, Tile *tile);
static void   _um_flags_update_turn(FlagsGame *game);

/*
 * Cancel the current AI task.
 */
static void _um_flags_ai_task_cancel(FlagsGame *game) {
    if (game->ai_task == PLAY) {
        game->ai_task = WAIT;
    }
}

/*
 * Set the AI task and, if it's an AI's turn, signal that it's ready.
 */
static void _um_flags_ai_task_set(FlagsGame *game) {
    if (game->state != DONE && game->turn_player >= game->params.num_hotseat) {
        game->ai_task = PLAY;
        pthread_cond_signal(&game->ai_task_ready);
    } else {
        game->ai_task = WAIT;
    }
}

/*
 * Check whether the AI thread should be started or stopped. Returns 1 on
 * error, otherwise 0.
 */
static int _um_flags_ai_thread_check(FlagsGame *game) {
    if (game->params.num_ai) {
        if (game->ai_state == DEAD) {
            return _um_flags_ai_thread_start(game);
        } else if (game->ai_task == DIE) {
            game->ai_task = WAIT;
        }
    } else if (!game->params.num_ai && game->ai_state == ALIVE) {
        _um_flags_ai_thread_stop(game, 0);
    }
    return 0;
}

/*
 * The AI thread. Waits on a condition variable for requests.
 */
static void * _um_flags_ai_thread_run(void *data) {
    FlagsGame *game = (FlagsGame *) data;
    pthread_mutex_lock(&game->global_mutex);
    while (game->ai_task != DIE) {
        while (game->ai_task == WAIT) {
            pthread_cond_wait(&game->ai_task_ready, &game->global_mutex);
        }
        if (game->ai_task == PLAY) {
            FlagsAIDecision decision = um_flags_ai_decide(
                game->players[game->turn_player].ai, game->tiles,
                game->params.width, game->params.height
            );
            if (decision.bomb) {
                _um_flags_bomb(game, game->turn_player, &decision.coords);
            } else {
                _um_flags_select(game, game->turn_player, &decision.coords);
            }
            pthread_mutex_unlock(&game->global_mutex);
            sleep(1);
            pthread_mutex_lock(&game->global_mutex);
        }
    }
    game->ai_state = DEAD;
    pthread_cond_signal(&game->ai_task_ready);
    pthread_mutex_unlock(&game->global_mutex);
    pthread_exit(NULL);
}

/*
 * Start the AI thread. Returns 1 on error, otherwise 0.
 */
static int _um_flags_ai_thread_start(FlagsGame *game) {
    pthread_attr_t attr;
    game->ai_task = WAIT;
    if (!pthread_attr_init(&attr)) {
        pthread_t thread;
        pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
        if (!pthread_create(&thread, &attr, _um_flags_ai_thread_run,
                                                            (void *) game)) {
            game->ai_state = ALIVE;
            pthread_attr_destroy(&attr);
            return 0;
        }
        pthread_attr_destroy(&attr);
    }
    return 1;
}

/*
 * Stop the AI thread, optionally waiting for it to finish.
 */
static void _um_flags_ai_thread_stop(FlagsGame *game, int wait) {
    game->ai_task = DIE;
    pthread_cond_signal(&game->ai_task_ready);
    if (wait) {
        while (game->ai_state == ALIVE) {
            pthread_cond_wait(&game->ai_task_ready, &game->global_mutex);
        }
    }
}

/*
 * See um_flags_bomb.
 */
static int _um_flags_bomb(FlagsGame *game, int player, const Coords *coords) {
    int expands = 0,
        hit = 0,
        range,
        width, height,
        i, j;
    if (game->state == DONE || game->turn_player != player ||
        !game->players[player].num_bombs) {
        return -1;
    }
    height = game->params.height;
    width = game->params.width;
    range = game->options.bomb_range;
    for (i = coords->x - range; i <= coords->x + range; ++i) {
        for (j = coords->y - range; j <= coords->y + range; ++j) {
            Coords reveal;
            int reveal_result;
            if (i < 0 || j < 0 || i >= width || j >= height ||
                game->tiles[i][j].revealed ||
                game->tiles[i][j].face >= PLAYER_FLAG) {
                continue;
            }
            reveal.x = i;
            reveal.y = j;
            reveal_result = _um_flags_reveal(game, player, &reveal);
            if (reveal_result == 1) {
                hit = 1;
            } else if (reveal_result == -1) {
                expands = 1;
            }
            if (!expands) {
                _um_flags_update_tile(game, &game->tiles[i][j]);
            }
        }
    }
    --game->players[player].num_bombs;
    return _um_flags_reveal_end(game, coords, expands, 1, hit, 1);
}

/*
 * Check for new AI levels, updating player data as required. Returns 1 if any
 * have changed, otherwise 0.
 */
static int _um_flags_check_ai_levels(FlagsGame *game) {
    int change = 0,
        num_hotseat = game->params.num_hotseat,
        i;
    for (i = 0; i < game->params.num_ai; ++i) {
        if (game->params.ai_levels[i] != game->players[num_hotseat + i].ai) {
            game->players[num_hotseat + i].ai = game->params.ai_levels[i];
            change = 1;
        }
    }
    return change;
}

/*
 * Set the game's state to DONE if the top player's score is so high (relative
 * to the number of remaining mines) that no player can catch up. Returns 1 if
 * the game is DONE, otherwise 0.
 */
static int _um_flags_check_win(FlagsGame *game) {
    int done = 0,
        top1 = 0,
        top2 = 0,
        total = 0,
        i;
    for (i = 0; i < game->params.num_hotseat + game->params.num_ai; ++i) {
        int score = game->players[i].score;
        total += score;
        if (score >= top1) {
            top2 = top1;
            top1 = score;
        } else if (score > top2) {
            top2 = score;
        }
    }
    if (game->options.allow_ties) {
        done = game->params.num_mines - total == 0 ||
               game->params.num_mines - total <  top1 - top2;
    } else {
        done = game->params.num_mines - total <= top1 - top2;
    }
    if (done) {
        game->state = DONE;
        _um_flags_update_state(game);
        if (game->params.num_ai) {
            _um_flags_ai_task_set(game);
        }
        if (!game->options.cover_on_done) {
            _um_flags_reveal_mined(game, 0);
        }
        return 1;
    }
    return 0;
}

/*
 * Create a new board, fetch its tiles, and mine it. Returns 1 on failure,
 * otherwise 0.
 */
static int _um_flags_new_board(FlagsGame *game) {
    game->board = um_board_create(game->params.width, game->params.height);
    if (!game->board) {
        return 1;
    }
    game->tiles = um_board_tiles(game->board);
    um_board_mine_random(game->board, game->params.num_mines, NULL, 0);
    game->mine_count = game->params.num_mines;
    return 0;
}

/*
 * Create a new players array and reset player data. Returns 1 on failure,
 * otherwise 0.
 */
static int _um_flags_new_players(FlagsGame *game) {
    game->players = malloc(
        sizeof(FlagsPlayer) * (game->params.num_hotseat + game->params.num_ai)
    );
    if (!game->players) {
        return 1;
    }
    _um_flags_reset_players(game);
    _um_flags_reset_ai_levels(game);
    return 0;
}

/*
 * Advance to the next turn.
 */
static void _um_flags_next_turn(FlagsGame *game) {
    ++game->turn_player;
    if (game->turn_player >= game->params.num_hotseat + game->params.num_ai) {
        game->turn_player = 0;
    }
    if (game->params.num_ai) {
        _um_flags_ai_task_set(game);
    }
    _um_flags_update_turn(game);
}

/*
 * Reset the board, player data, state, turn, and AI task. This does nothing
 * if the game is already NOT_STARTED. Specifying skip_board or skip_players
 * skips resetting the board or player data.
 */
static void _um_flags_reset_all(FlagsGame *game, int skip_board,
                                int skip_players) {
    if (game->state == NOT_STARTED) {
        return;
    }
    if (!skip_board) {
        _um_flags_reset_board(game);
        _um_flags_update_board(game);
        _um_flags_update_mine_count(game);
    }
    if (!skip_players) {
        _um_flags_reset_players(game);
        _um_flags_update_players(game);
    }
    if (game->params.num_ai) {
        _um_flags_ai_task_cancel(game);
    }
    game->state = NOT_STARTED;
    game->turn_player = 0;
    _um_flags_update_state(game);
    _um_flags_update_turn(game);
}

/*
 * Reset AI levels.
 */
static void _um_flags_reset_ai_levels(FlagsGame *game) {
    int num_hotseat = game->params.num_hotseat,
        num_players = num_hotseat + game->params.num_ai,
        i;
    for (i = 0; i < num_players; ++i) {
        if (i < num_hotseat) {
            game->players[i].ai = NO_AI;
        } else {
            game->players[i].ai = game->params.ai_levels[i - num_hotseat];
        }
    }
}

/*
 * Clear the board and mine it.
 */
static void _um_flags_reset_board(FlagsGame *game) {
    um_board_clear(game->board);
    um_board_mine_random(game->board, game->params.num_mines, NULL, 0);
    game->mine_count = game->params.num_mines;
}

/*
 * Reset player scores and bomb counts.
 */
static void _um_flags_reset_players(FlagsGame *game) {
    FlagsPlayer *players = game->players;
    int num_bombs = game->options.num_bombs,
        num_players = game->params.num_hotseat + game->params.num_ai,
        i;
    for (i = 0; i < num_players; ++i) {
        players[i].num_bombs = num_bombs;
        players[i].score = 0;
    }
}

/*
 * Reveal the tile at coords for player. Returns 0 if a tile update is required
 * (the tile isn't mined, and is either non-zero or no_expand_zero_tiles is
 * on), 1 if the tile is mined (a player update is required, and win conditions
 * must be checked), and -1 if a board update is required (the tile is a zero
 * tile and no_expand_zero_tiles is off).
 */
static int _um_flags_reveal(FlagsGame *game, int player,
                            const Coords *coords) {
    int reveal_result = um_board_reveal(game->board, coords);
    if (reveal_result == -1) {
        ++game->players[player].score;
        --game->mine_count;
        game->tiles[coords->x][coords->y].face = PLAYER_FLAG + player;
        game->tiles[coords->x][coords->y].revealed = 0;
        return 1;
    } else if (!reveal_result && !game->options.no_expand_zero_tiles) {
        um_board_expand(game->board, coords);
        return -1;
    }
    return 0;
}

/*
 * End the turn after one or more reveals (check win conditions after a hit,
 * update the board or tile, and advance to the next turn without a hit).
 * Returns 0 if it's still the player's turn, otherwise 1.
 */
static int _um_flags_reveal_end(FlagsGame *game, const Coords *coords,
                                int update_board, int update_players,
                                int hit, int bomb) {
    int return_value = 1,
        win = 0;
    if (game->state == NOT_STARTED) {
        game->state = IN_PROGRESS;
        _um_flags_update_state(game);
    }
    if (update_players || hit) {
        _um_flags_update_mine_count(game);
        _um_flags_update_players(game);
    }
    if (hit) {
        win = _um_flags_check_win(game);
        if (win) {
            if (!game->options.cover_on_done) {
                update_board = 1;
            }
        } else if (!bomb || !game->options.bomb_ends_turn) {
            return_value = 0;
        }
    }
    if (update_board) {
        _um_flags_update_board(game);
    } else {
        _um_flags_update_tile(game, &game->tiles[coords->x][coords->y]);
         /* TODO this update is issued redundantly when a bomb has no expands
          * and doesn't win the game; fix is uncertain */
    }
    if (return_value && !win) {
        _um_flags_next_turn(game);
    }
    return return_value;
}

/*
 * Reveal (or cover) unflagged mines.
 */
static void _um_flags_reveal_mined(FlagsGame *game, int cover) {
    int i, j,
        new_val = 1;
    if (cover) {
        new_val = 0;
    }
    for (i = 0; i < game->params.width; i += 1) {
        for (j = 0; j < game->params.height; j += 1) {
            if (game->tiles[i][j].mined &&
                game->tiles[i][j].face < PLAYER_FLAG) {
                game->tiles[i][j].revealed = new_val;
            }
        }
    }
}

/*
 * See um_flags_select.
 */
static int _um_flags_select(FlagsGame *game, int player,
                            const Coords *coords) {
    int reveal;
    if ((game->state == NOT_STARTED && player) || game->state == DONE ||
        game->turn_player != player ||
        game->tiles[coords->x][coords->y].revealed ||
        game->tiles[coords->x][coords->y].face >= PLAYER_FLAG) {
        return -1;
    }
    reveal = _um_flags_reveal(game, player, coords);
    return _um_flags_reveal_end(game, coords, reveal == -1, 0, reveal == 1, 0);
}

/*
 * Transition the game between on and off states for the allow_ties option.
 */
static void _um_flags_transition_allow_ties(FlagsGame *game, int old,
                                            int new) {
    if (game->state == IN_PROGRESS) {
        if (old && !new) {
            _um_flags_check_win(game);
        } else if (!old && new) {
            /* No action required. */
        }
    }
}

/*
 * Transition the game between on and off states for the cover_on_done option.
 */
static void _um_flags_transition_cover_on_done(FlagsGame *game, int old,
                                                int new) {
    if (game->state == DONE) {
        if (old && !new) {
            _um_flags_reveal_mined(game, 0);
        } else if (!old && new) {
            _um_flags_reveal_mined(game, 1);
        }
        _um_flags_update_board(game);
    }
}

/*
 * Transition the game between on and off states for the no_expand_zero_tiles
 * option.
 */
static void _um_flags_transition_no_expand_zero_tiles(FlagsGame *game, int old,
                                                        int new) {
    if (old && !new) {
        int change = 0,
            i, j;
        for (i = 0; i < game->params.width; ++i) {
            for (j = 0; j < game->params.height; ++j) {
                if (!game->tiles[i][j].count && game->tiles[i][j].revealed &&
                    !game->tiles[i][j].mined) {
                    Coords coords;
                    coords.x = i;
                    coords.y = j;
                    um_board_expand(game->board, &coords);
                    change = 1;
                }
            }
        }
        if (change) {
            _um_flags_update_board(game);
        }
    } else if (!old && new) {
        /* No action required; it's not possible to un-expand zero tiles. */
    }
}

/*
 * Transition the game between on and off states for the num_bombs option.
 */
static void _um_flags_transition_num_bombs(FlagsGame *game, int new) {
    int num_players = game->params.num_hotseat + game->params.num_ai,
        i;
    for (i = 0; i < num_players; ++i) {
        game->players[i].num_bombs = new;
    }
    _um_flags_update_players(game);
}

/*
 * Call the user's update_board callback, if provided.
 */
static void _um_flags_update_board(FlagsGame *game) {
    if (game->callbacks.update_board) {
        game->callbacks.update_board(game->tiles, game->callbacks.data);
    }
}

/*
 * Call the user's update_mine_count callback, if provided.
 */
static void _um_flags_update_mine_count(FlagsGame *game) {
    if (game->callbacks.update_mine_count) {
        game->callbacks.update_mine_count(game->mine_count,
                                            game->callbacks.data);
    }
}

/*
 * Call the user's update_players callback, if provided.
 */
static void _um_flags_update_players(FlagsGame *game) {
    if (game->callbacks.update_players) {
        game->callbacks.update_players(
            game->players,
            game->params.num_hotseat + game->params.num_ai,
            game->callbacks.data
        );
    }
}

/*
 * Call the user's update_state callback, if provided.
 */
static void _um_flags_update_state(FlagsGame *game) {
    if (game->callbacks.update_state) {
        game->callbacks.update_state(game->state, game->callbacks.data);
    }
}

/*
 * Call the user's update_tile callback with the given tile, if provided.
 */
static void _um_flags_update_tile(FlagsGame *game, Tile *tile) {
    if (game->callbacks.update_tile) {
        game->callbacks.update_tile(tile, game->callbacks.data);
    }
}

/*
 * Call the user's update_turn callback, if provided.
 */
static void _um_flags_update_turn(FlagsGame *game) {
    if (game->callbacks.update_turn) {
        game->callbacks.update_turn(game->turn_player, game->callbacks.data);
    }
}

/*
 * Public functions.
 */

/*
 * Create a new game with the given callbacks, options, and parameters. Returns
 * a pointer to the new game, or NULL. If the game is created with AI players,
 * then a new thread is started to compute and apply AI decisions, and all
 * callbacks must be thread-safe.
 */
FlagsGame * um_flags_create(const FlagsCallbacks *callbacks,
                            const FlagsOptions *options,
                            const FlagsParameters *params) {
    FlagsGame *game;
    game = malloc(sizeof(FlagsGame));
    if (game) {
        game->params = *params;
        if (!_um_flags_new_board(game)) {
            game->options = *options;
            if (!_um_flags_new_players(game)) {
                if (!pthread_mutex_init(&game->global_mutex, NULL)) {
                    if (!pthread_cond_init(&game->ai_task_ready, NULL)) {
                        game->ai_state = DEAD;
                        if (!_um_flags_ai_thread_check(game)) {
                            game->callbacks = *callbacks;
                            game->state = NOT_STARTED;
                            game->turn_player = 0;
                            return game;
                        }
                        pthread_cond_destroy(&game->ai_task_ready);
                    }
                    pthread_mutex_destroy(&game->global_mutex);
                }
                free(game->players);
            }
            um_board_destroy(game->board);
        }
        free(game);
    }
    return NULL;
}

/*
 * Destroy the game. If there are AI players, this blocks until the AI thread
 * terminates.
 */
void um_flags_destroy(FlagsGame *game) {
    pthread_mutex_lock(&game->global_mutex);
    if (game->params.num_ai) {
        _um_flags_ai_thread_stop(game, 1);
    }
    pthread_mutex_unlock(&game->global_mutex);
    pthread_cond_destroy(&game->ai_task_ready);
    pthread_mutex_destroy(&game->global_mutex);
    if (game->players) {
        free(game->players);
    }
    if (game->board) {
        um_board_destroy(game->board);
    }
    free(game);
}

/*
 * Calls your callback functions, allowing your interface to refresh when the
 * game hasn't changed. You should rarely need to use this.
 */
void um_flags_update(FlagsGame *game) {
    pthread_mutex_lock(&game->global_mutex);
    _um_flags_update_board(game);
    _um_flags_update_mine_count(game);
    _um_flags_update_players(game);
    _um_flags_update_state(game);
    if (game->state != DONE) {
        _um_flags_update_turn(game);
    }
    pthread_mutex_unlock(&game->global_mutex);
}

/*
 * Set the game's options. Returns -1 if the player is not the game master,
 * otherwise 0. Changing the allow ties option may declare the game DONE,
 * changing the no_expand_zero_tiles option may reveal one or more tiles, and
 * changing the cover_on_done option will reveal or cover tiles if the game is
 * DONE. This does not reset the game.
 */
int um_flags_options(FlagsGame *game, int player,
                        const FlagsOptions *options) {
    FlagsOptions old_opts;
    if (player) {
        return -1;
    }
    pthread_mutex_lock(&game->global_mutex);
    old_opts = game->options;
    game->options = *options;
    if (old_opts.allow_ties != options->allow_ties) {
        _um_flags_transition_allow_ties(game, old_opts.allow_ties,
                                        options->allow_ties);
    }
    if (old_opts.cover_on_done != options->cover_on_done) {
        _um_flags_transition_cover_on_done(game, old_opts.cover_on_done,
                                            options->cover_on_done);
    }
    if (old_opts.no_expand_zero_tiles != options->no_expand_zero_tiles) {
        _um_flags_transition_no_expand_zero_tiles(
            game, old_opts.no_expand_zero_tiles, options->no_expand_zero_tiles
        );
    }
    if (old_opts.num_bombs != options->num_bombs) {
        _um_flags_transition_num_bombs(game, options->num_bombs);
    }
    pthread_mutex_unlock(&game->global_mutex);
    return 0;
}

/*
 * Set the size of the game's board and/or the number of mines or players,
 * resetting the game. Returns -1 if the player is not the game master, 1 if an
 * error occurs and the game must be destroyed, otherwise 0. If the game has
 * only hotseat players and the new parameters add AI players, then a new
 * thread is started to compute and apply AI decisions, and all callbacks must
 * be thread-safe. If the new parameters remove all AI players, then the AI
 * thread is terminated (this does not block).
 */
int um_flags_parameters(FlagsGame *game, int player,
                        const FlagsParameters *params) {
    FlagsParameters old_params;
    int skip_board = 0,
        skip_players = 0;
    if (player) {
        return -1;
    }
    pthread_mutex_lock(&game->global_mutex);
    old_params = game->params;
    game->params = *params;
    if (old_params.width != params->width ||
        old_params.height != params->height) {
        um_board_destroy(game->board);
        if (_um_flags_new_board(game)) {
            game->params.num_ai = old_params.num_ai;
            pthread_mutex_unlock(&game->global_mutex);
            return 1;
        }
        _um_flags_update_board(game);
        _um_flags_update_mine_count(game);
        skip_board = 1;
    } else if (old_params.num_mines != params->num_mines) {
        _um_flags_reset_board(game);
        _um_flags_update_board(game);
        _um_flags_update_mine_count(game);
        skip_board = 1;
    }
    if (old_params.num_hotseat != params->num_hotseat ||
        old_params.num_ai != params->num_ai) {
        if (old_params.num_hotseat + old_params.num_ai !=
            params->num_hotseat + params->num_ai) {
            free(game->players);
            if (_um_flags_new_players(game)) {
                game->params.num_ai = old_params.num_ai;
                pthread_mutex_unlock(&game->global_mutex);
                return 1;
            }
        } else {
            _um_flags_reset_players(game);
            _um_flags_reset_ai_levels(game);
        }
        if (old_params.num_ai != params->num_ai) {
            _um_flags_ai_thread_check(game);
        }
        _um_flags_update_players(game);
        skip_players = 1;
    } else if (_um_flags_check_ai_levels(game)) {
        _um_flags_update_players(game);
    }
    _um_flags_reset_all(game, skip_board, skip_players);
    pthread_mutex_unlock(&game->global_mutex);
    return 0;
}

/*
 * Reset the game. Returns -1 if the player is not the game master,
 * otherwise 0.
 */
int um_flags_reset(FlagsGame *game, int player) {
    if (player) {
        return -1;
    }
    pthread_mutex_lock(&game->global_mutex);
    _um_flags_reset_all(game, 0, 0);
    pthread_mutex_unlock(&game->global_mutex);
    return 0;
}

/*
 * Select the tile at the given coords. Returns -1 if it's not the player's
 * turn, the game is DONE, or the tile is already revealed, 0 if it's still the
 * player's turn, otherwise 1. This may reveal one or more other tiles if the
 * no_expand_zero_tiles option is off, or declare the game DONE.
 */
int um_flags_select(FlagsGame *game, int player, const Coords *coords) {
    int select;
    pthread_mutex_lock(&game->global_mutex);
    select = _um_flags_select(game, player, coords);
    pthread_mutex_unlock(&game->global_mutex);
    return select;
}

/*
 * Select the tiles in the square centred at centre and extending bomb_range
 * tiles in each direction. Returns -1 if it's not the player's turn, the
 * player has no bombs left, or the game is DONE, 0 if it's still the player's
 * turn, otherwise 1. This may reveal one or more other tiles if the
 * no_expand_zero_tiles option is off, or declare the game DONE.
 */
int um_flags_bomb(FlagsGame *game, int player, const Coords *coords) {
    int result;
    pthread_mutex_lock(&game->global_mutex);
    result = _um_flags_bomb(game, player, coords);
    pthread_mutex_unlock(&game->global_mutex);
    return result;
}
