/*
 * unmine/flags.h
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef UM_FLAGS_
#define UM_FLAGS_

#include "coords.h"
#include "flags_ai.h"
#include "tile.h"

/*
 * Types.
 */

typedef struct FlagsGame FlagsGame;

typedef enum {
    BLANK, PLAYER_FLAG
} FlagsFace;

typedef enum {
    NOT_STARTED, IN_PROGRESS, DONE, NUM_FLAGS_STATES
} FlagsState;

typedef struct {
    int allow_ties, bomb_ends_turn, bomb_range, cover_on_done,
        no_expand_zero_tiles, num_bombs;
} FlagsOptions;

typedef struct {
    int width, height, num_mines, num_hotseat, num_ai;
    FlagsAILevel *ai_levels;
} FlagsParameters;

typedef struct {
    int num_bombs, score;
    FlagsAILevel ai;
} FlagsPlayer;

typedef struct {
    void (*update_board)(Tile **tiles, void *data);
    void (*update_mine_count)(int mine_count, void *data);
    void (*update_players)(const FlagsPlayer *players, int count, void *data);
    void (*update_state)(FlagsState state, void *data);
    void (*update_tile)(const Tile *tile, void *data);
    void (*update_turn)(int player, void *data);
    void *data;
} FlagsCallbacks;

/*
 * Constants.
 */

extern const FlagsParameters UM_FLAGS_MIN;
extern const FlagsParameters UM_FLAGS_DEFAULT;
extern const FlagsParameters UM_FLAGS_MAX;

/*
 * Functions.
 */

/*
 * Create a new game with the given callbacks, options, and parameters. Returns
 * a pointer to the new game, or NULL. If the game is created with AI players,
 * then a new thread is started to compute and apply AI decisions, and all
 * callbacks must be thread-safe.
 */
FlagsGame * um_flags_create(const FlagsCallbacks *callbacks,
                            const FlagsOptions *options,
                            const FlagsParameters *params);

/*
 * Destroy the game. If there are AI players, this blocks until the AI thread
 * terminates.
 */
void um_flags_destroy(FlagsGame *game);

/*
 * Calls your callback functions, allowing your interface to refresh when the
 * game hasn't changed. You should rarely need to use this.
 */
void um_flags_update(FlagsGame *game);

/*
 * Set the game's options. Returns -1 if the player is not the game master,
 * otherwise 0. Changing the allow ties option may declare the game DONE,
 * changing the no_expand_zero_tiles option may reveal one or more tiles, and
 * changing the cover_on_done option will reveal or cover tiles if the game is
 * DONE. This does not reset the game.
 */
int um_flags_options(FlagsGame *game, int player, const FlagsOptions *options);

/*
 * Set the size of the game's board and/or the number of mines or players,
 * resetting the game. Returns -1 if the player is not the game master, 1 if an
 * error occurs and the game must be destroyed, otherwise 0. If the game has
 * only hotseat players and the new parameters add AI players, then a new
 * thread is started to compute and apply AI decisions, and all callbacks must
 * be thread-safe. If the new parameters remove all AI players, then the AI
 * thread is terminated (this does not block).
 */
int um_flags_parameters(FlagsGame *game, int player,
                        const FlagsParameters *params);

/*
 * Reset the game. Returns -1 if the player is not the game master,
 * otherwise 0.
 */
int um_flags_reset(FlagsGame *game, int player);

/*
 * Select the tile at the given coords. Returns -1 if it's not the player's
 * turn, the game is DONE, or the tile is already revealed, 0 if it's still the
 * player's turn, otherwise 1. This may reveal one or more other tiles if the
 * no_expand_zero_tiles option is off, or declare the game DONE.
 */
int um_flags_select(FlagsGame *game, int player, const Coords *coords);

/*
 * Select the tiles in the square centred at centre and extending bomb_range
 * tiles in each direction. Returns -1 if it's not the player's turn, the
 * player has no bombs left, or the game is DONE, 0 if it's still the player's
 * turn, otherwise 1. This may reveal one or more other tiles if the
 * no_expand_zero_tiles option is off, or declare the game DONE.
 */
int um_flags_bomb(FlagsGame *game, int player, const Coords *coords);

#endif /* UM_FLAGS_ */
