/*
 * unmine/flags_ai.c
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "flags_ai.h"

#include <assert.h>
#include <stdlib.h>

#include "flags.h"

/* Strategies   RANDOM          selects random coords
 *              ODDS            selects the tile most likely to be a mine
 *              CAUTIOUS        selects the safest move
 *
 * Skill Levels NOVICE          SB, OM,         RA-ON
 *              JUNIOR          SB, OM, OC,     RA-NN
 *              INTERMEDIATE        OM, OC, NM, RA-NN
 *              ADVANCED            OM, EC, NM, PM
 *              MASTER              OM, EC, NM, SM
 *
 * SB       sometimes uses the bomb at random (likelihood varies)
 * OM       selects obvious mines
 * OC       bombs obvious clusters
 * EC       bombs only to win the game
 * NM       selects non-obvious mines
 * RA-ON    selects randomly, avoiding obvious non-mines
 * RA-NN    like RA-ON, but also avoids non-obvious non-mines
 * PM       selects the most probable mine
 * SM       like PM, but avoids zero tiles
 *
 * TODO: an end-game situation makes safety irrelevant; can we do more than EC?
 */

/*
 * Constants.
 */

const FlagsAILevel UM_FLAGS_AI_DEFAULT = NOVICE;

/*
 * Strings.
 */

const char * const UM_FLAGS_AI_LEVEL_NAMES[] = {
    "None",
    "Random",
    "Odds",
    "Cautious",
    "Novice",
    "Junior",
    "Intermediate",
    "Advanced",
    "Master"
};

/*
 * Types.
 */

typedef struct {
    int p_mined,    /* estimated probability of being mined /100, or -1 */
        p_safe;     /* estimated probability of being safe /100, or -1 */
} TileKnowledge;

/*
 * Private functions.
 */

/*
 * Returns a random integer between 0 and n-1.
 */
static int _um_flags_ai_random_int(int n) {
    return (int) ((double) rand() / ((double) RAND_MAX + 1) * n);
}

/*
 * Detect obvious mines. This detects when a revealed, non-mined tile's count
 * equals the number of (revealed) neighbouring mines plus the number of
 * unrevealed neighbours, minus the number of unrevealed neighbours known not
 * to be mines (knowledge is optional). If a mine is found, the coords are
 * written to the given decision and this returns 0. Otherwise, returns 1.
 */
static int _um_flags_ai_find_mines_easy(Tile **tiles, TileKnowledge **know,
                                        int width, int height,
                                        FlagsAIDecision *decision) {
    int i;
    for (i = 0; i < width; ++i) {
        int j;
        for (j = 0; j < height; ++j) {
            Tile *neighbours[8];
            int count,
                r = 0,
                unrevealed = 0,
                k;
            if (!tiles[i][j].revealed ||
                tiles[i][j].mined ||
                !tiles[i][j].count) {
                continue;
            }
            /* This tile is displaying a count. */
            count = tiles[i][j].count;
            um_tile_neighbours(tiles, &tiles[i][j], width, height, neighbours);
            for (k = 0; k < 8; ++k) {
                if (!neighbours[k]) {
                    continue;
                }
                if (neighbours[k]->face >= PLAYER_FLAG) {
                    --count;
                    if (count < 1) {
                        break;
                    }
                } else if (!neighbours[k]->revealed &&
                           (!know ||
                            know[neighbours[k]->x][neighbours[k]->y].p_mined)) {
                    ++unrevealed;
                }
            }
            /* Neighbouring this tile are count unrevealed mines, and
             * unrevealed unrevealed tiles that could be mines. */
            if (count < 1 || count != unrevealed) {
                continue;
            }
            /* These quantities are equal, so all those tiles are mines. */
            if (count > 1) {
                r = _um_flags_ai_random_int(count);
            }
            for (k = 0; k < 8; ++k) {
                if (!neighbours[k]) {
                    continue;
                }
                if (neighbours[k]->face < PLAYER_FLAG &&
                    !neighbours[k]->revealed &&
                    (!know ||
                     know[neighbours[k]->x][neighbours[k]->y].p_mined)) {
                    --r;
                    if (r < 0) {
                        assert(neighbours[k]->mined);
                        decision->coords.x = neighbours[k]->x;
                        decision->coords.y = neighbours[k]->y;
                        return 0;
                    }
                }
            }
        }
    }
    return 1;
}

/*
 * Detect non-obvious mines. This detects when a revealed, non-mined tile T has
 * unrevealed mined neighbours and neighbouring tile V has even more. If the
 * number of unrevealed mines neighbouring V, subtract the same for T, equals
 * the number of unrevealed tiles neighbouring V but not T, subtract the number
 * of those known not to be mined, then the others are mines (knowledge is
 * optional). If a mine is found, the coords are written to the given decision
 * and this returns 0. Otherwise, returns 1.
 */
static int _um_flags_ai_find_mines_hard(Tile **tiles, TileKnowledge **know,
                                        int width, int height,
                                        FlagsAIDecision *decision) {
    int i;
    for (i = 0; i < width; ++i) {
        int j;
        for (j = 0; j < height; ++j) {
            Tile *neighbours[8];
            int count,
                k;
            if (!tiles[i][j].revealed ||
                tiles[i][j].mined ||
                !tiles[i][j].count) {
                continue;
            }
            /* This is tile T. It's displaying a count. */
            count = tiles[i][j].count;
            um_tile_neighbours(tiles, &tiles[i][j], width, height, neighbours);
            for (k = 0; k < 8; ++k) {
                if (!neighbours[k]) {
                    continue;
                }
                if (neighbours[k]->face >= PLAYER_FLAG) {
                    --count;
                    if (count < 1) {
                        break;
                    }
                }
            }
            if (count < 1) {
                continue;
            }
            /* There are count unrevealed mines neighbouring tile T. */
            for (k = 0; k < 8; ++k) {
                Tile *neighbours2[8];
                int n2_only[8] = {0},
                    only_count = 0,
                    count2,
                    m,
                    r;
                if (!neighbours[k]) {
                    continue;
                }
                if (!neighbours[k]->revealed ||
                    neighbours[k]->mined ||
                    !neighbours[k]->count) {
                    continue;
                }
                /* This is tile V, neighbour of T, also displaying a count. */
                count2 = neighbours[k]->count;
                um_tile_neighbours(tiles, neighbours[k], width, height,
                                    neighbours2);
                for (m = 0; m < 8; ++m) {
                    if (!neighbours2[m]) {
                        continue;
                    }
                    if (neighbours2[m]->face >= PLAYER_FLAG) {
                        --count2;
                        if (count2 <= count) {
                            break;
                        }
                    } else if (!neighbours2[m]->revealed &&
                               (!know ||
                                know[neighbours2[m]->x][neighbours2[m]->y].p_mined)) {
                        if (!um_tile_is_neighbour(neighbours2[m], &tiles[i][j])) {
                            n2_only[m] = 1;
                            ++only_count;
                        }
                    }
                }
                /* There are count2 unrevealed mines neighbouring tile V, and
                 * only_count tiles neighbouring V but not T that could be
                 * mines. */
                if (count2 <= count || count2 - count != only_count) {
                    continue;
                }
                /* Since count2 > count, there are count2 - count unrevealed
                 * mines neighbouring V but not T. Since
                 * count2 - count == only_count, all those tiles are mined. */
                r = 0;
                if (only_count > 1) {
                    r = _um_flags_ai_random_int(only_count);
                }
                for (m = 0; m < 8; ++m) {
                    if (n2_only[m]) {
                        --r;
                        if (r < 0) {
                            assert(neighbours2[m]->mined);
                            decision->coords.x = neighbours2[m]->x;
                            decision->coords.y = neighbours2[m]->y;
                            return 0;
                        }
                    }
                }
            }
        }
    }
    return 1;
}

/*
 * Detect obvious non-mines. This detects when a revealed, non-mined tile's
 * count equals the number of revealed, mined neighbours, and marks any
 * unrevealed neighbours as non-mined. Returns 0 if knowledge is gained.
 * Otherwise, returns 1.
 */
static int _um_flags_ai_find_nonmines_easy(Tile **tiles, TileKnowledge **know,
                                            int width, int height) {
    int changed = 0,
        i;
    for (i = 0; i < width; ++i) {
        int j;
        for (j = 0; j < height; ++j) {
            Tile *neighbours[8];
            int count,
                unrevealed = 0,
                k;
            if (!tiles[i][j].revealed ||
                tiles[i][j].mined ||
                !tiles[i][j].count) {
                continue;
            }
            /* This tile is displaying a count. */
            count = tiles[i][j].count;
            um_tile_neighbours(tiles, &tiles[i][j], width, height, neighbours);
            for (k = 0; k < 8; ++k) {
                if (!neighbours[k]) {
                    continue;
                }
                if (neighbours[k]->face >= PLAYER_FLAG) {
                    --count;
                } else if (!neighbours[k]->revealed &&
                           know[neighbours[k]->x][neighbours[k]->y].p_mined) {
                    unrevealed = 1;
                }
            }
            /* Neighbouring this tile are count unrevealed mines and, if
             * unrevealed, at least one unrevealed tile. */
            if (count || !unrevealed) {
                continue;
            }
            /* All mines neighbouring this tile are revealed, but some
             * neighbours aren't. Those tiles are obviously not mines. */
            for (k = 0; k < 8; ++k) {
                if (!neighbours[k]) {
                    continue;
                }
                if (neighbours[k]->face < PLAYER_FLAG &&
                    !neighbours[k]->revealed &&
                    know[neighbours[k]->x][neighbours[k]->y].p_mined) {
                    know[neighbours[k]->x][neighbours[k]->y].p_mined = 0;
                    assert(!neighbours[k]->mined);
                    changed = 1;
                }
            }
        }
    }
    return !changed;
}

/*
 * Detect non-obvious non-mines. This detects two possible scenarios occuring
 * when a tile T and a neighbouring tile V are both revealed, non-mined, and
 * have unrevealed neighbouring mines. In the first scenario, the number of
 * unrevealed mines neighbouring T equals the number neighbouring V. If V's
 * unrevealed neighbours contain T's, then unrevealed neighbours of V but not T
 * are non-mines. In the second scenario, there are more unrevealed mines
 * neighbouring V than T. If their difference equals the number of unrevealed
 * tiles neighbouring V but not T, then unrevealed neighbours of T but not V
 * are non-mines. These scenarios are generalizations of the '1-1' and '1-2'
 * patterns, respectively. TODO: I'm unsure either is fully generalized.
 */
static int _um_flags_ai_find_nonmines_hard(Tile **tiles, TileKnowledge **know,
                                            int width, int height) {
    int changed = 0,
        i;
    for (i = 0; i < width; ++i) {
        int j;
        for (j = 0; j < height; ++j) {
            Tile *neighbours[8];
            int count,
                k;
            if (!tiles[i][j].revealed ||
                tiles[i][j].mined ||
                !tiles[i][j].count) {
                continue;
            }
            /* This tile is displaying a count. */
            count = tiles[i][j].count;
            um_tile_neighbours(tiles, &tiles[i][j], width, height, neighbours);
            for (k = 0; k < 8; ++k) {
                if (!neighbours[k]) {
                    continue;
                }
                if (neighbours[k]->face >= PLAYER_FLAG) {
                    --count;
                    if (count < 1) {
                        break;
                    }
                }
            }
            if (count < 1) {
                continue;
            }
            /* There are count unrevealed mines neighbouring tile T. */
            for (k = 0; k < 8; ++k) {
                Tile *neighbours2[8];
                int count2,
                    m;
                if (!neighbours[k]) {
                    continue;
                }
                if (!neighbours[k]->revealed ||
                    neighbours[k]->mined ||
                    !neighbours[k]->count) {
                    continue;
                }
                /* This is tile V, neighbour of T, also displaying a count. */
                count2 = neighbours[k]->count;
                um_tile_neighbours(tiles, neighbours[k], width, height,
                                    neighbours2);
                for (m = 0; m < 8; ++m) {
                    if (!neighbours2[m]) {
                        continue;
                    }
                    if (neighbours2[m]->face >= PLAYER_FLAG) {
                        --count2;
                        if (count2 < count) {
                            break;
                        }
                    }
                }
                /* There are count2 unrevealed mines neighbouring tile V. */
                if (count2 < count) {
                    continue;
                }
                if (count == count2) {
                    /* This is pattern 1. */
                    int contained = 1;
                    for (m = 0; m < 8; ++m) {
                        if (!neighbours[m]) {
                            continue;
                        }
                        if (neighbours[m]->face >= PLAYER_FLAG ||
                            neighbours[m]->revealed ||
                            !know[neighbours[m]->x][neighbours[m]->y].p_mined) {
                            continue;
                        }
                        if (!um_tile_is_neighbour(neighbours[m],
                                                    neighbours[k])) {
                            contained = 0;
                            break;
                        }
                    }
                    if (!contained) {
                        continue;
                    }
                    /* All unrevealed neighbours of T are neighbours of V. Any
                     * unrevealed neighbours of V that don't neighbour T are
                     * non-mines. */
                    for (m = 0; m < 8; ++m) {
                        if (!neighbours2[m]) {
                            continue;
                        }
                        if (neighbours2[m]->face >= PLAYER_FLAG ||
                            neighbours2[m]->revealed ||
                            !know[neighbours2[m]->x][neighbours2[m]->y].p_mined) {
                            continue;
                        }
                        if (!um_tile_is_neighbour(neighbours2[m], &tiles[i][j])) {
                            know[neighbours2[m]->x][neighbours2[m]->y].p_mined = 0;
                            assert(!neighbours2[m]->mined);
                            changed = 1;
                        }
                    }
                } else {
                    /* This is pattern 2. */
                    int count3 = 0;
                    for (m = 0; m < 8; ++m) {
                        if (!neighbours2[m]) {
                            continue;
                        }
                        if (neighbours2[m]->face >= PLAYER_FLAG ||
                            neighbours2[m]->revealed ||
                            !know[neighbours2[m]->x][neighbours2[m]->y].p_mined) {
                            continue;
                        }
                        if (!um_tile_is_neighbour(neighbours2[m],
                                                    &tiles[i][j])) {
                            ++count3;
                        }
                    }
                    /* There are count3 unrevealed tiles neighbouring V but not
                     * T. */
                    if (count2 - count != count3) {
                        continue;
                    }
                    /* There are count2 - count unrevealed mines neighbouring V
                     * but not T. Since count2 - count == count3, those tiles
                     * are mines and the unrevealed tiles neighbouring T but
                     * not V are non-mines. */
                    for (m = 0; m < 8; ++m) {
                        if (!neighbours[m]) {
                            continue;
                        }
                        if (neighbours[m]->face >= PLAYER_FLAG ||
                            neighbours[m]->revealed ||
                            !know[neighbours[m]->x][neighbours[m]->y].p_mined) {
                            continue;
                        }
                        if (!um_tile_is_neighbour(neighbours[m],
                                                    neighbours[k])) {
                            know[neighbours[m]->x][neighbours[m]->y].p_mined = 0;
                            assert(!neighbours[m]->mined);
                            changed = 1;
                        }
                    }
                }
            }
        }
    }
    return !changed;
}

/*
 * Mark tiles neighbouring mines, and tiles whose neighbours are revealed, as
 * safe. Returns 0 if knowledge is gained. Otherwise, returns 1.
 */
static int _um_flags_ai_find_safe(Tile **tiles, TileKnowledge **know,
                                    int width, int height) {
    int changed = 0,
        i;
    for (i = 0; i < width; ++i) {
        int j;
        for (j = 0; j < height; ++j) {
            Tile *neighbours[8];
            int k;
            if (tiles[i][j].revealed) {
                continue;
            }
            /* This tile is either unrevealed, or a revealed mine. */
            um_tile_neighbours(tiles, &tiles[i][j], width, height, neighbours);
            if (tiles[i][j].face >= PLAYER_FLAG) {
                /* This tile is a revealed mine; its neighbours are safe. */
                for (k = 0; k < 8; ++k) {
                    if (!neighbours[k]) {
                        continue;
                    }
                    if (neighbours[k]->face < PLAYER_FLAG &&
                        !neighbours[k]->revealed &&
                        know[neighbours[k]->x][neighbours[k]->y].p_safe != 100) {
                        know[neighbours[k]->x][neighbours[k]->y].p_safe = 100;
                        assert(neighbours[k]->count);
                        changed = 1;
                    }
                }
            } else {
                /* This tile is unrevealed. Check if its neighbours are too. */
                int neighbours_revealed = 1;
                for (k = 0; k < 8; ++k) {
                    if (neighbours[k] &&
                        !neighbours[k]->revealed &&
                        neighbours[k]->face < PLAYER_FLAG) {
                        neighbours_revealed = 0;
                        break;
                    }
                }
                /* Mark the tile safe if all of its neighbours are revealed. */
                if (neighbours_revealed && know[i][j].p_safe != 100) {
                    know[i][j].p_safe = 100;
                    changed = 1;
                }
            }
        }
    }
    return !changed;
}

/*
 * This function uses the given knowledge to make a guess (knowledge is
 * optional). If possible, this selects a safe tile not known to be non-mined.
 * If no such tile exists, and prefer_safe is specified, this selects a safe
 * tile. If a safe tile doesn't exist, or prefer_safe is not specified, this
 * selects a tile not known to be non-mined. If any such tiles have known
 * probabilities, only a tile with the highest probability will be selected.
 */
static FlagsAIDecision _um_flags_ai_guess(Tile **tiles, TileKnowledge **know,
                                            int width, int height,
                                            int prefer_safe) {
    FlagsAIDecision decision = {{0, 0}, 0};
    int count = 0,
        max_prob = 0,
        only_mined = 0,
        only_safe = 0,
        r = 0,
        i;
    if (know) {
        int exists_safe = 0,
            exists_both = 0,
            x;
        /* Check for tiles that are safe, possibly mined, or both. */
        for (x = 0; x < width; ++x) {
            int y;
            for (y = 0; y < height; ++y) {
                if (tiles[x][y].face >= PLAYER_FLAG || tiles[x][y].revealed) {
                    continue;
                }
                if (know[x][y].p_safe == 100) {
                    exists_safe = 1;
                    if (know[x][y].p_mined != 0) {
                        exists_both = 1;
                        break;
                    }
                }
            }
            if (exists_both) {
                break;
            }
        }
        /* Enable restrictions based on available tiles and prefer_safe. */
        if (exists_both) {
            only_mined = 1;
            only_safe = 1;
        } else if (exists_safe && prefer_safe) {
            only_safe = 1;
        } else {
            only_mined = 1;
        }
        /* Find the maximum known probability. */
        for (x = 0; x < width; ++x) {
            int y;
            for (y = 0; y < height; ++y) {
                if (tiles[x][y].face >= PLAYER_FLAG ||
                    tiles[x][y].revealed ||
                    (only_mined && know[x][y].p_mined == 0) ||
                    (only_safe && know[x][y].p_safe < 100)) {
                    continue;
                }
                if (know[x][y].p_mined > max_prob) {
                    max_prob = know[x][y].p_mined;
                }
            }
        }
    }
    /* Find the number of candidate tiles. */
    for (i = 0; i < width; ++i) {
        int j;
        for (j = 0; j < height; ++j) {
            if (tiles[i][j].face >= PLAYER_FLAG ||
                tiles[i][j].revealed ||
                (only_mined && know[i][j].p_mined == 0) ||
                (only_safe && know[i][j].p_safe < 100) ||
                (max_prob && know[i][j].p_mined < max_prob)) {
                continue;
            }
            ++count;
        }
    }
    /* Select a candidate tile at random. */
    if (count > 1) {
        r = _um_flags_ai_random_int(count);
    }
    for (i = 0; i < width; ++i) {
        int j;
        for (j = 0; j < height; ++j) {
            if (tiles[i][j].face >= PLAYER_FLAG ||
                tiles[i][j].revealed ||
                (only_mined && know[i][j].p_mined == 0) ||
                (only_safe && know[i][j].p_safe < 100) ||
                (max_prob && know[i][j].p_mined < max_prob)) {
                continue;
            }
            --r;
            if (r < 0) {
                decision.coords.x = i;
                decision.coords.y = j;
                break;
            }
        }
        if (r < 0) {
            break;
        }
    }
    return decision;
}

/*
 * Create a knowledge array of the given dimensions.
 */
static TileKnowledge ** _um_flags_ai_knowledge_create(int width, int height) {
    int i;
    TileKnowledge **pointers, *knowledge;
    pointers = malloc(sizeof(TileKnowledge *) * width);
    if (!pointers) {
        return NULL;
    }
    knowledge = malloc(sizeof(TileKnowledge) * width * height);
    if (!knowledge) {
        free(pointers);
        return NULL;
    }
    for (i = 0; i < width; ++i) {
        int j;
        pointers[i] = &knowledge[i * height];
        for (j = 0; j < height; ++j) {
            pointers[i][j].p_mined = -1;
            pointers[i][j].p_safe = -1;
        }
    }
    return pointers;
}

/*
 * Destroy the knowledge array.
 */
static void _um_flags_ai_knowledge_destroy(TileKnowledge **knowledge) {
    free(&knowledge[0][0]);
    free(knowledge);
}

/*
 * Decide based on the given skill level. As the skill level increases, this
 * function uses progressively better mine, non-mine, and safe tile detection,
 * and probability calculation.
 */
static FlagsAIDecision _um_flags_ai_skill_level(FlagsAILevel level,
                                                Tile **tiles, int width,
                                                int height) {
    FlagsAIDecision decision = {{0, 0}, 0};
    TileKnowledge **know;
    int prefer_safe = 0;
    if (level < INTERMEDIATE) {
        /* TODO: bomb at random, relative to level */
    }
    if (!_um_flags_ai_find_mines_easy(tiles, NULL, width, height, &decision)) {
        return decision;
    }
    know = _um_flags_ai_knowledge_create(width, height);
    if (!know) {
        return _um_flags_ai_guess(tiles, NULL, width, height, 0);
    }
    _um_flags_ai_find_nonmines_easy(tiles, know, width, height);
    if (level > NOVICE) {
        if (!_um_flags_ai_find_mines_easy(tiles, know, width, height,
                                            &decision)) {
            _um_flags_ai_knowledge_destroy(know);
            return decision;
        }
        if (level < ADVANCED) {
            /* TODO: bomb any obvious cluster */
        }
        _um_flags_ai_find_nonmines_hard(tiles, know, width, height);
        if (level > JUNIOR) {
            if (!_um_flags_ai_find_mines_hard(tiles, know, width, height,
                                                &decision)) {
                _um_flags_ai_knowledge_destroy(know);
                return decision;
            }
            if (level > INTERMEDIATE) {
                /* TODO: bomb an end-game cluster */
                /* TODO: add probabilities to knowledge */
                if (level > ADVANCED) {
                    _um_flags_ai_find_safe(tiles, know, width, height);
                    prefer_safe = 1;
                }
            }
        }
    }
    decision = _um_flags_ai_guess(tiles, know, width, height, prefer_safe);
    _um_flags_ai_knowledge_destroy(know);
    return decision;
}

/*
 * Select the tile most likely to be mined, optionally selecting only safe
 * tiles.
 */
static FlagsAIDecision _um_flags_ai_strategy_odds(Tile **tiles, int width,
                                                    int height, int safe) {
    TileKnowledge **know = _um_flags_ai_knowledge_create(width, height);
    if (know) {
        FlagsAIDecision guess;
        if (safe) {
            _um_flags_ai_find_safe(tiles, know, width, height);
        }
        /* TODO: add probabilities to knowledge */
        guess = _um_flags_ai_guess(tiles, know, width, height, safe);
        _um_flags_ai_knowledge_destroy(know);
        return guess;
    }
    return _um_flags_ai_guess(tiles, NULL, width, height, 0);
}

/*
 * Select a random unrevealed tile.
 */
static FlagsAIDecision _um_flags_ai_strategy_random(Tile **tiles, int width,
                                                    int height) {
    return _um_flags_ai_guess(tiles, NULL, width, height, 0);
}

/*
 * Public functions.
 */

/*
 * Decide a player's next action based on the given skill level or strategy.
 * TODO: to detect end-game bomb situations, this function needs the number of
 * remaining bombs and the scores of all players.
 */
FlagsAIDecision um_flags_ai_decide(FlagsAILevel level, Tile **tiles, int width,
                                    int height) {
    if (level == RANDOM) {
        return _um_flags_ai_strategy_random(tiles, width, height);
    } else if (level == ODDS) {
        return _um_flags_ai_strategy_odds(tiles, width, height, 0);
    } else if (level == CAUTIOUS) {
        return _um_flags_ai_strategy_odds(tiles, width, height, 1);
    }
    return _um_flags_ai_skill_level(level, tiles, width, height);
}
