/*
 * unmine/flags_config.c
 *
 * Copyright 2017 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#define _POSIX_C_SOURCE 199506L

#include "flags_config.h"

#include <strings.h>

#include "flags_ai.h"

/*
 * Strings.
 */

static const char *NAME_ALLOW_TIES = "allow_ties";
static const char *NAME_BOMB_ENDS_TURN = "bomb_ends_turn";
static const char *NAME_BOMB_RANGE = "bomb_range";
static const char *NAME_COVER_ON_DONE = "cover_on_done";
static const char *NAME_NO_EXPAND_ZERO_TILES = "no_expand_zero_tiles";
static const char *NAME_NUM_BOMBS = "num_bombs";

static const char *NAME_WIDTH = "width";
static const char *NAME_HEIGHT = "height";
static const char *NAME_NUM_MINES = "num_mines";
static const char *NAME_NUM_HOTSEAT = "num_hotseat";
static const char *NAME_NUM_AI = "num_ai";
static const char *NAME_AI_LEVELS = "ai_levels";

/*
 * Public functions.
 */

/*
 * Add configuration settings for flags game options to the given group. This
 * can be called repeatedly with the same group to populate missing settings.
 */
void um_flags_config_create_options(config_setting_t *group) {
    if (!config_setting_get_member(group, NAME_ALLOW_TIES)) {
        config_setting_add(group, NAME_ALLOW_TIES, CONFIG_TYPE_BOOL);
    }
    if (!config_setting_get_member(group, NAME_BOMB_ENDS_TURN)) {
        config_setting_add(group, NAME_BOMB_ENDS_TURN, CONFIG_TYPE_BOOL);
    }
    if (!config_setting_get_member(group, NAME_BOMB_RANGE)) {
        config_setting_add(group, NAME_BOMB_RANGE, CONFIG_TYPE_INT);
    }
    if (!config_setting_get_member(group, NAME_COVER_ON_DONE)) {
        config_setting_add(group, NAME_COVER_ON_DONE, CONFIG_TYPE_BOOL);
    }
    if (!config_setting_get_member(group, NAME_NO_EXPAND_ZERO_TILES)) {
        config_setting_add(group, NAME_NO_EXPAND_ZERO_TILES, CONFIG_TYPE_BOOL);
    }
    if (!config_setting_get_member(group, NAME_NUM_BOMBS)) {
        config_setting_add(group, NAME_NUM_BOMBS, CONFIG_TYPE_INT);
    }
}

/*
 * Add configuration settings for flags game parameters to the given group.
 * This can be called repeatedly with the same group to populate missing
 * settings.
 */
void um_flags_config_create_parameters(config_setting_t *group) {
    if (!config_setting_get_member(group, NAME_WIDTH)) {
        config_setting_add(group, NAME_WIDTH, CONFIG_TYPE_INT);
    }
    if (!config_setting_get_member(group, NAME_HEIGHT)) {
        config_setting_add(group, NAME_HEIGHT, CONFIG_TYPE_INT);
    }
    if (!config_setting_get_member(group, NAME_NUM_MINES)) {
        config_setting_add(group, NAME_NUM_MINES, CONFIG_TYPE_INT);
    }
    if (!config_setting_get_member(group, NAME_NUM_HOTSEAT)) {
        config_setting_add(group, NAME_NUM_HOTSEAT, CONFIG_TYPE_INT);
    }
    if (!config_setting_get_member(group, NAME_NUM_AI)) {
        config_setting_add(group, NAME_NUM_AI, CONFIG_TYPE_INT);
    }
    if (!config_setting_get_member(group, NAME_AI_LEVELS)) {
        config_setting_add(group, NAME_AI_LEVELS, CONFIG_TYPE_ARRAY);
    }
}

/*
 * Convenience function; calls functions um_flags_config_load_*, and calls
 * functions um_flags_config_create_* to fill in missing settings.
 */
void um_flags_config_load(config_setting_t *group, FlagsOptions *options,
                            FlagsParameters *params, int max_players) {
    if (um_flags_config_load_options(group, options)) {
        um_flags_config_create_options(group);
    }
    if (um_flags_config_load_parameters(group, params, max_players)) {
        um_flags_config_create_parameters(group);
    }
}

/*
 * Load flags game options from the configuration group. No changes are made to
 * options with missing or invalid settings. Returns 1 if one or more settings
 * are missing or invalid, otherwise 0.
 */
int um_flags_config_load_options(config_setting_t *group,
                                    FlagsOptions *options) {
    int bomb_range = 0,
        num_bombs = -1,
        result_at,
        result_bet,
        result_br,
        result_cod,
        result_nezt,
        result_nb;
    result_at = config_setting_lookup_bool(group, NAME_ALLOW_TIES,
                                            &options->allow_ties);
    result_bet = config_setting_lookup_bool(group, NAME_BOMB_ENDS_TURN,
                                            &options->bomb_ends_turn);
    result_br = config_setting_lookup_int(group, NAME_BOMB_RANGE, &bomb_range);
    if (result_br == CONFIG_TRUE &&
        bomb_range > 0 &&
        bomb_range < UM_FLAGS_MAX.width) { /* TODO: a proper limit. */
        options->bomb_range = bomb_range;
    }
    result_cod = config_setting_lookup_bool(group, NAME_COVER_ON_DONE,
                                            &options->cover_on_done);
    result_nezt = config_setting_lookup_bool(group, NAME_NO_EXPAND_ZERO_TILES,
                                            &options->no_expand_zero_tiles);
    result_nb = config_setting_lookup_int(group, NAME_NUM_BOMBS, &num_bombs);
    if (result_nb == CONFIG_TRUE &&
        num_bombs >= 0 &&
        num_bombs < UM_FLAGS_MAX.width) { /* TODO: a proper limit. */
        options->num_bombs = num_bombs;
    }
    return result_at == CONFIG_FALSE ||
           result_bet == CONFIG_FALSE ||
           result_br == CONFIG_FALSE ||
           result_cod == CONFIG_FALSE ||
           result_nezt == CONFIG_FALSE ||
           result_nb == CONFIG_FALSE;
}

/*
 * Load flags game parameters from the configuration group. No changes are made
 * to parameters with missing or invalid settings. Up to max_players AI levels
 * and players will be loaded. Returns 1 if one or more settings are missing or
 * invalid, otherwise 0.
 */
int um_flags_config_load_parameters(config_setting_t *group,
                                    FlagsParameters *params, int max_players) {
    config_setting_t *setting_al;
    int result_w,
        result_h,
        result_nm,
        result_nh,
        result_na,
        result_al = 0,
        width = 0,
        height = 0,
        num_mines = 0,
        num_hotseat = -1,
        num_ai = -1;
    result_w = config_setting_lookup_int(group, NAME_WIDTH, &width);
    if (result_w == CONFIG_TRUE &&
        width >= UM_FLAGS_MIN.width &&
        width <= UM_FLAGS_MAX.width) {
        params->width = width;
    }
    result_h = config_setting_lookup_int(group, NAME_HEIGHT, &height);
    if (result_h == CONFIG_TRUE &&
        height >= UM_FLAGS_MIN.height &&
        height <= UM_FLAGS_MAX.height) {
        params->height = height;
    }
    result_nm = config_setting_lookup_int(group, NAME_NUM_MINES, &num_mines);
    if (result_nm == CONFIG_TRUE &&
        num_mines >= UM_FLAGS_MIN.num_mines &&
        num_mines <= UM_FLAGS_MAX.num_mines &&
        num_mines < (width * height - 2)) {
        params->num_mines = num_mines;
    }
    result_nh = config_setting_lookup_int(group, NAME_NUM_HOTSEAT, &num_hotseat);
    if (result_nh == CONFIG_TRUE &&
        num_hotseat >= UM_FLAGS_MIN.num_hotseat &&
        num_hotseat <= UM_FLAGS_MAX.num_hotseat) {
        params->num_hotseat = num_hotseat;
    }
    if (params->num_hotseat > max_players) {
        params->num_hotseat = max_players;
    }
    if (params->num_ai > max_players) {
        params->num_ai = max_players;
    }
    if (params->num_hotseat + params->num_ai > max_players) {
        params->num_ai = max_players - params->num_hotseat;
    }
    result_na = config_setting_lookup_int(group, NAME_NUM_AI, &num_ai);
    if (result_na == CONFIG_TRUE &&
        num_ai >= UM_FLAGS_MIN.num_ai &&
        num_ai <= UM_FLAGS_MAX.num_ai) {
        params->num_ai = num_ai;
    }
    setting_al = config_setting_get_member(group, NAME_AI_LEVELS);
    if (setting_al && config_setting_is_array(setting_al)) {
        int i;
        result_al = 1;
        for (i = 0; i < max_players; ++i) {
            const char *level = config_setting_get_string_elem(setting_al, i);
            int j;
            if (!level) {
                break;
            }
            for (j = NO_AI + 1; j < NUM_AI_LEVELS; ++j) {
                if (!strcasecmp(level, UM_FLAGS_AI_LEVEL_NAMES[j])) {
                    params->ai_levels[i] = j;
                    break;
                }
            }
        }
    }
    return result_w == CONFIG_FALSE ||
           result_h == CONFIG_FALSE ||
           result_nm == CONFIG_FALSE ||
           result_nh == CONFIG_FALSE ||
           result_na == CONFIG_FALSE ||
           !result_al;
}

/*
 * Convenience function; calls functions um_flags_config_save_*.
 */
void um_flags_config_save(config_setting_t *group, FlagsOptions *options,
                            FlagsParameters *params, int max_levels) {
    um_flags_config_save_options(group, options);
    um_flags_config_save_parameters(group, params, max_levels);
}

/*
 * Save flags game options into the configuration group. No changes are made to
 * missing or invalid settings. Returns 1 if one or more settings are missing
 * or invalid, otherwise 0.
 */
int um_flags_config_save_options(config_setting_t *group,
                                    FlagsOptions *options) {
    config_setting_t *setting_at,
                     *setting_bet,
                     *setting_br,
                     *setting_cod,
                     *setting_nezt,
                     *setting_nb;
    int result_at = CONFIG_FALSE,
        result_bet = CONFIG_FALSE,
        result_br = CONFIG_FALSE,
        result_cod = CONFIG_FALSE,
        result_nezt = CONFIG_FALSE,
        result_nb = CONFIG_FALSE;
    setting_at = config_setting_get_member(group, NAME_ALLOW_TIES);
    if (setting_at) {
        result_at = config_setting_set_bool(setting_at, options->allow_ties);
    }
    setting_bet = config_setting_get_member(group, NAME_BOMB_ENDS_TURN);
    if (setting_bet) {
        result_bet = config_setting_set_bool(setting_bet,
                                                options->bomb_ends_turn);
    }
    setting_br = config_setting_get_member(group, NAME_BOMB_RANGE);
    if (setting_br) {
        result_br = config_setting_set_int(setting_br, options->bomb_range);
    }
    setting_cod = config_setting_get_member(group, NAME_COVER_ON_DONE);
    if (setting_cod) {
        result_cod = config_setting_set_bool(setting_cod,
                                                options->cover_on_done);
    }
    setting_nezt = config_setting_get_member(group, NAME_NO_EXPAND_ZERO_TILES);
    if (setting_nezt) {
        result_nezt = config_setting_set_bool(setting_nezt,
                                                options->no_expand_zero_tiles);
    }
    setting_nb = config_setting_get_member(group, NAME_NUM_BOMBS);
    if (setting_nb) {
        result_nb = config_setting_set_int(setting_nb, options->num_bombs);
    }
    return result_at == CONFIG_FALSE ||
           result_bet == CONFIG_FALSE ||
           result_br == CONFIG_FALSE ||
           result_cod == CONFIG_FALSE ||
           result_nezt == CONFIG_FALSE ||
           result_nb == CONFIG_FALSE;
}

/*
 * Save flags game parameters into the configuration group. No changes are made
 * to missing or invalid settings. Up to max_levels AI levels will be saved.
 * Returns 1 if one or more settings are missing or invalid, otherwise 0.
 */
int um_flags_config_save_parameters(config_setting_t *group,
                                    FlagsParameters *params, int max_levels) {
    config_setting_t *setting_w,
                     *setting_h,
                     *setting_nm,
                     *setting_nh,
                     *setting_na,
                     *setting_al;
    int result_w = CONFIG_FALSE,
        result_h = CONFIG_FALSE,
        result_nm = CONFIG_FALSE,
        result_nh = CONFIG_FALSE,
        result_na = CONFIG_FALSE,
        result_al = 0;
    setting_w = config_setting_get_member(group, NAME_WIDTH);
    if (setting_w) {
        result_w = config_setting_set_int(setting_w, params->width);
    }
    setting_h = config_setting_get_member(group, NAME_HEIGHT);
    if (setting_h) {
        result_h = config_setting_set_int(setting_h, params->height);
    }
    setting_nm = config_setting_get_member(group, NAME_NUM_MINES);
    if (setting_nm) {
        result_nm = config_setting_set_int(setting_nm, params->num_mines);
    }
    setting_nh = config_setting_get_member(group, NAME_NUM_HOTSEAT);
    if (setting_nh) {
        result_nh = config_setting_set_int(setting_nh, params->num_hotseat);
    }
    setting_na = config_setting_get_member(group, NAME_NUM_AI);
    if (setting_na) {
        result_na = config_setting_set_int(setting_na, params->num_ai);
    }
    setting_al = config_setting_get_member(group, NAME_AI_LEVELS);
    if (setting_al && config_setting_is_array(setting_al)) {
        int count = config_setting_length(setting_al),
            i;
        while (count < max_levels) {
            config_setting_add(setting_al, NULL, CONFIG_TYPE_STRING);
            ++count;
        }
        for (i = 0; i < max_levels; ++i) {
            config_setting_set_string_elem(setting_al, i,
                                UM_FLAGS_AI_LEVEL_NAMES[params->ai_levels[i]]);
        }
    } else {
        result_al = 1;
    }
    return result_w == CONFIG_FALSE ||
           result_h == CONFIG_FALSE ||
           result_nm == CONFIG_FALSE ||
           result_nh == CONFIG_FALSE ||
           result_na == CONFIG_FALSE ||
           result_al == 1;
}
