/*
 * unmine/gui.c
 *
 * Copyright 2017 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "gui.h"

#include <stdlib.h>
#include <time.h>

#include <gtk/gtk.h>
#include <libconfig.h>

#include "gui_classic.h"
#include "gui_flags.h"
#include "unmine.h"

/*
 * Strings.
 */

static const char *CONFIG_FILE = ".unmine-gui";
static const char *CONFIG_NAME_CLASSIC = "classic";
static const char *CONFIG_NAME_FLAGS = "flags";
static const char *CONFIG_NAME_GAME = "game";
static const char *CONFIG_NAME_MAXIMIZED = "maximized";
static const char *MENU_GAME = "_Game";
static const char *MENU_GAME_CLASSIC = "_Classic";
static const char *MENU_GAME_FLAGS = "_Flags";
static const char *MENU_GAME_ABOUT = "_About";
static const char *MENU_GAME_EXIT = "E_xit";
static const char *PATH_ICON = "icons/64x64/flag.png";

/*
 * Widgets.
 */

static GdkPixbuf *icon;
static GtkWidget *classic_box,
                 *classic_menu,
                 *flags_box,
                 *flags_menu,
                 *window;

/*
 * Private functions.
 */

/* Initialization. */
static GtkWidget * _um_gui_build_game_menu(config_setting_t *config_game);
static GtkWidget * _um_gui_build_menu(GUIClassic *classic, GUIFlags *flags,
                                        config_setting_t *game);
static GtkWidget * _um_gui_build_toplevel(config_setting_t *maximized);
static void        _um_gui_build(GtkWidget *window, GUIClassic *classic,
                                    GUIFlags *flags, config_setting_t *game);

/* Configuration. */
static void        _um_gui_config_init(config_t *config);
static void        _um_gui_config_destroy(config_t *config);

/* UI actions. */
static void        _um_gui_switch_classic(void);
static void        _um_gui_switch_flags(void);

/* Signal handlers. */
static void        _um_gui_about(void);
static void        _um_gui_game_classic(GtkWidget *menu_item,
                                        config_setting_t *game);
static void        _um_gui_game_flags(GtkWidget *menu_item,
                                        config_setting_t *game);
static int         _um_gui_save_maximized(config_setting_t *maximized,
                                            GdkEventWindowState *event);

/*
 * Build the 'Game' (main) drop-down menu.
 */
static GtkWidget * _um_gui_build_game_menu(config_setting_t *config_game) {
    GtkWidget *about,
              *active_game,
              *classic,
              *exit,
              *flags,
              *game,
              *game_menu,
              *sep_game;

    /* Create menu items, setting their options as necessary. */
    game = gtk_menu_item_new_with_mnemonic(MENU_GAME);
    classic = gtk_radio_menu_item_new_with_mnemonic(NULL, MENU_GAME_CLASSIC);
    flags = gtk_radio_menu_item_new_with_mnemonic_from_widget(
                GTK_RADIO_MENU_ITEM(classic), MENU_GAME_FLAGS);
    if (config_setting_get_int(config_game)) {
        active_game = flags;
    } else {
        active_game = classic;
    }
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(active_game), TRUE);
    sep_game = gtk_separator_menu_item_new();
    about = gtk_menu_item_new_with_mnemonic(MENU_GAME_ABOUT);
    exit = gtk_menu_item_new_with_mnemonic(MENU_GAME_EXIT);

    /* Create menus and fill them with menu items. */
    game_menu = gtk_menu_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(game_menu), classic);
    gtk_menu_shell_append(GTK_MENU_SHELL(game_menu), flags);
    gtk_menu_shell_append(GTK_MENU_SHELL(game_menu), sep_game);
    gtk_menu_shell_append(GTK_MENU_SHELL(game_menu), about);
    gtk_menu_shell_append(GTK_MENU_SHELL(game_menu), exit);

    /* Link menu items to submenus. */
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(game), game_menu);

    /* Connect signal handlers. */
    g_signal_connect(classic, "activate",
                        G_CALLBACK(_um_gui_game_classic), config_game);
    g_signal_connect(flags, "activate",
                        G_CALLBACK(_um_gui_game_flags), config_game);
    g_signal_connect(about, "activate", G_CALLBACK(_um_gui_about), NULL);
    g_signal_connect(exit, "activate", G_CALLBACK(gtk_main_quit), NULL);

    /* Return root menu item. */
    return game;
}

/*
 * Build the menu bar.
 */
static GtkWidget * _um_gui_build_menu(GUIClassic *classic, GUIFlags *flags,
                                        config_setting_t *game) {
    GtkWidget *menu_bar;
    classic_menu = um_gui_classic_get_menu(classic);
    flags_menu = um_gui_flags_get_menu(flags);
    menu_bar = gtk_menu_bar_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(menu_bar),
                            _um_gui_build_game_menu(game));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu_bar), classic_menu);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu_bar), flags_menu);
    return menu_bar;
}

/*
 * Build the top-level window.
 */
static GtkWidget * _um_gui_build_toplevel(config_setting_t *maximized) {
    GtkWidget *top = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(top), UM_NAME);
    gtk_container_set_border_width(GTK_CONTAINER(top), 0);
    if (config_setting_get_bool(maximized)) {
        gtk_window_maximize(GTK_WINDOW(top));
    }
    g_signal_connect(top, "destroy", G_CALLBACK(gtk_main_quit), NULL);
    g_signal_connect_swapped(top, "window-state-event",
                                G_CALLBACK(_um_gui_save_maximized), maximized);
    return top;
}

/*
 * Build the GUI with the given top-level GtkWindow and game UIs.
 */
static void _um_gui_build(GtkWidget *top, GUIClassic *classic,
                            GUIFlags *flags, config_setting_t *game) {
    GtkWidget *vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);
    gtk_container_add(
        GTK_CONTAINER(vbox),
        _um_gui_build_menu(classic, flags, game)
    );
    classic_box = um_gui_classic_get_ui(classic);
    gtk_box_pack_start(GTK_BOX(vbox), classic_box, 1, 1, 0);
    flags_box = um_gui_flags_get_ui(flags);
    gtk_box_pack_start(GTK_BOX(vbox), flags_box, 1, 1, 0);
    gtk_container_add(GTK_CONTAINER(top), vbox);
    icon = gdk_pixbuf_new_from_file(PATH_ICON, NULL);
    gtk_window_set_default_icon(icon);
    gtk_widget_show_all(top);
    if (config_setting_get_int(game)) {
        _um_gui_switch_flags();
    } else {
        _um_gui_switch_classic();
    }
}

/*
 * Initialize the config object, read the config file, and add missing options.
 */
static void _um_gui_config_init(config_t *config) {
    config_setting_t *root;
    config_init(config);
    config_read_file(config, CONFIG_FILE);
    root = config_root_setting(config);
    if (!config_setting_get_member(root, CONFIG_NAME_GAME)) {
        config_setting_add(root, CONFIG_NAME_GAME, CONFIG_TYPE_INT);
    }
    if (!config_setting_get_member(root, CONFIG_NAME_MAXIMIZED)) {
        config_setting_add(root, CONFIG_NAME_MAXIMIZED, CONFIG_TYPE_BOOL);
    }
    if (!config_setting_get_member(root, CONFIG_NAME_CLASSIC)) {
        config_setting_add(root, CONFIG_NAME_CLASSIC, CONFIG_TYPE_GROUP);
    }
    if (!config_setting_get_member(root, CONFIG_NAME_FLAGS)) {
        config_setting_add(root, CONFIG_NAME_FLAGS, CONFIG_TYPE_GROUP);
    }
}

/*
 * Save the config file and destroy the config object.
 */
static void _um_gui_config_destroy(config_t *config) {
    config_write_file(config, CONFIG_FILE);
    config_destroy(config);
}

/*
 * Switch to the Classic game.
 */
static void _um_gui_switch_classic() {
    gtk_widget_hide(flags_box);
    gtk_widget_hide(flags_menu);
    gtk_widget_show(classic_box);
    gtk_widget_show(classic_menu);
    um_gui_shrink();
}

/*
 * Switch to the Flags game.
 */
static void _um_gui_switch_flags() {
    gtk_widget_hide(classic_box);
    gtk_widget_hide(classic_menu);
    gtk_widget_show(flags_box);
    gtk_widget_show(flags_menu);
    um_gui_shrink();
}

/*
 * Run the 'About' dialog. Signal handler for the 'About' menu item.
 */
static void _um_gui_about() {
    gtk_show_about_dialog(
        GTK_WINDOW(window),
        "program-name", UM_NAME,
        "version", UM_VERSION,
        "copyright", UM_COPYRIGHT,
        "license-type", GTK_LICENSE_GPL_2_0,
        NULL
    );
}

/*
 * Signal handler for the 'Classic' menu item. If active, switch to the Classic
 * game and save the game setting.
 */
static void _um_gui_game_classic(GtkWidget *menu_item,
                                    config_setting_t *game) {
    if (gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menu_item))) {
        _um_gui_switch_classic();
        config_setting_set_int(game, 0);
    }
}

/*
 * Signal handler for the 'Flags' menu item. If active, switch to the Flags
 * game and save the game setting.
 */
static void _um_gui_game_flags(GtkWidget *menu_item, config_setting_t *game) {
    if (gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menu_item))) {
        _um_gui_switch_flags();
        config_setting_set_int(game, 1);
    }
}

/*
 * Save the current maximized state to the configuration setting. Signal
 * handler for the toplevel window.
 */
static int _um_gui_save_maximized(config_setting_t *maximized,
                                    GdkEventWindowState *event) {
    config_setting_set_bool(
        maximized,
        (event->new_window_state & GDK_WINDOW_STATE_MAXIMIZED) != 0
    );
    return GDK_EVENT_PROPAGATE;
}

/*
 * Public functions.
 */

/*
 * Start the interface.
 */
int main(int argc, char **argv) {
    config_t config;
    config_setting_t *root;
    GUIClassic *classic;
    GUIFlags *flags;
    srand(time(NULL));
    if (!gtk_init_check(&argc, &argv)) {
        return 1;
    }
    _um_gui_config_init(&config);
    root = config_root_setting(&config);
    window = _um_gui_build_toplevel(
                    config_setting_get_member(root, CONFIG_NAME_MAXIMIZED));
    classic = um_gui_classic_init(window,
                        config_setting_get_member(root, CONFIG_NAME_CLASSIC));
    if (!classic) {
        gtk_widget_destroy(window);
        config_destroy(&config);
        return 1;
    }
    flags = um_gui_flags_init(window,
                        config_setting_get_member(root, CONFIG_NAME_FLAGS));
    if (!flags) {
        um_gui_classic_destroy(classic);
        gtk_widget_destroy(window);
        config_destroy(&config);
        return 1;
    }
    _um_gui_build(window, classic, flags,
                    config_setting_get_member(root, CONFIG_NAME_GAME));
    gtk_main();
    um_gui_classic_destroy(classic);
    um_gui_flags_destroy(flags);
    _um_gui_config_destroy(&config);
    g_object_unref(icon);
    return 0;
}

/*
 * Shrink the main window to its contents.
 */
void um_gui_shrink(void) {
    gtk_window_resize(GTK_WINDOW(window), 1, 1);
}
