/*
 * unmine/gui_board.c
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

/*
 * TODO: feature: (option to) highlight tiles based on the last property
 *
 * TODO: ensure everything is functional with a keyboard
 *
 * TODO: check whether to explicitly set events with gtk_widget_set_events
 *
 */

#include "gui_board.h"

#include <stdlib.h>

/*
 * Strings.
 */
static const char *CHILD_BUTTON = "button";
static const char *CHILD_IMAGE = "image";
static const char *CHILD_LABEL = "label";
static const char * const MARKUP_NUMBER[] = {
    "<tt> </tt>",
    "<tt><span color=\"#0000ff\">1</span></tt>",
    "<tt><span color=\"#008000\">2</span></tt>",
    "<tt><span color=\"#ff0000\">3</span></tt>",
    "<tt><span color=\"#000080\">4</span></tt>",
    "<tt><span color=\"#800000\">5</span></tt>",
    "<tt><span color=\"#008080\">6</span></tt>",
    "<tt><span color=\"#000000\">7</span></tt>",
    "<tt><span color=\"#808080\">8</span></tt>"
};

/*
 * Types.
 */

struct GUIBoard {
    Coords active;
    GdkPixbuf **face_pixbufs,
              **mine_pixbufs;
    GtkWidget *widget;
    GUIBoardCallbacks callbacks;
    int **revealed,
        **face;
    Tile **tiles;
    int width,
        height;
};

/*
 * Private functions.
 */

/* Widget construction. */
static GtkWidget * _um_gui_board_build(GUIBoard *gboard);
static GtkWidget * _um_gui_board_build_button(GUIBoard *gboard);
static GtkWidget * _um_gui_board_build_tile(GUIBoard *gboard);
static void _um_gui_board_connect_mouse(GUIBoard *gboard, GtkWidget *widget);

/* Signal handlers. */
static int  _um_gui_board_enter(GtkWidget *widget, GdkEvent *event,
                                GUIBoard *gboard);
static int  _um_gui_board_leave(GtkWidget *widget, GdkEvent *event,
                                GUIBoard *gboard);
static int  _um_gui_board_press(GUIBoard *gboard, GdkEvent *event);
static int  _um_gui_board_release(GUIBoard *gboard, GdkEvent *event);
static void _um_gui_board_toggle(GtkWidget *widget, GUIBoard *gboard);

/* Other functions. */
static int  _um_gui_board_resize(GUIBoard *gboard, int width, int height);
static int  _um_gui_board_state_create(GUIBoard *gboard, int width,
                                        int height);
static void _um_gui_board_state_destroy(GUIBoard *gboard);
static void _um_gui_board_tile_coords(const GUIBoard *gboard,
                                        GtkWidget *widget, Coords *coords);
static void _um_gui_board_toggle_range(const GUIBoard *gboard, int depressed,
                                        int face, int i_start, int i_end,
                                        int j_start, int j_end);

/*
 * Build the board widget. Returns a GtkGrid.
 */
static GtkWidget * _um_gui_board_build(GUIBoard *gboard) {
    GtkWidget *widget = gtk_grid_new();
    gtk_grid_set_column_homogeneous(GTK_GRID(widget), 1);
    gtk_grid_set_row_homogeneous(GTK_GRID(widget), 1);
    gtk_widget_set_halign(widget, GTK_ALIGN_CENTER);
    g_signal_connect_swapped(widget, "button-press-event",
                                G_CALLBACK(_um_gui_board_press), gboard);
    g_signal_connect_swapped(widget, "button-release-event",
                                G_CALLBACK(_um_gui_board_release), gboard);
    return widget;
}

/*
 * Build a tile widget's button. Returns a GtkToggleButton.
 */
static GtkWidget * _um_gui_board_build_button(GUIBoard *gboard) {
    GtkWidget *button = gtk_toggle_button_new();
    gtk_container_add(GTK_CONTAINER(button), gtk_image_new());
    g_signal_connect(button, "toggled", G_CALLBACK(_um_gui_board_toggle),
                        gboard);
    _um_gui_board_connect_mouse(gboard, button);
    return button;
}

/*
 * Build a tile widget. Returns a GtkStack with children: GtkToggleButton,
 * GtkImage, and GtkLabel.
 */
static GtkWidget * _um_gui_board_build_tile(GUIBoard *gboard) {
    GtkWidget *stack = gtk_stack_new();
    gtk_stack_add_named(GTK_STACK(stack), _um_gui_board_build_button(gboard),
                        CHILD_BUTTON);
    gtk_stack_add_named(GTK_STACK(stack), gtk_image_new(), CHILD_IMAGE);
    gtk_stack_add_named(GTK_STACK(stack), gtk_label_new(NULL), CHILD_LABEL);
    gtk_widget_add_events(stack,
                            GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);
    _um_gui_board_connect_mouse(gboard, stack);
    return stack;
}

/*
 * Connect enter-notify-event and leave-notify-event signal handlers to the
 * given widget.
 */
static void _um_gui_board_connect_mouse(GUIBoard *gboard, GtkWidget *widget) {
    g_signal_connect(widget, "enter-notify-event",
                        G_CALLBACK(_um_gui_board_enter), gboard);
    g_signal_connect(widget, "leave-notify-event",
                        G_CALLBACK(_um_gui_board_leave), gboard);
}

/*
 * Set the active coords to those of the tile widget entered and call the
 * user-supplied enter callback. Signal handler for the enter-notify-event.
 */
static int _um_gui_board_enter(GtkWidget *widget, GdkEvent *event,
                                GUIBoard *gboard) {
    Coords coords;
    _um_gui_board_tile_coords(gboard, widget, &coords);
    gboard->active = coords;
    if (gboard->callbacks.enter) {
        gboard->callbacks.enter(&coords, gboard->callbacks.data);
    }
    return 1;
}

/*
 * Clear the active coords and call the user-supplied leave callback. Signal
 * handler for the leave-notify-event.
 */
static int _um_gui_board_leave(GtkWidget *widget, GdkEvent *event,
                                GUIBoard *gboard) {
    Coords coords;
    _um_gui_board_tile_coords(gboard, widget, &coords);
    gboard->active.x = -1;
    gboard->active.y = -1;
    if (gboard->callbacks.leave) {
        gboard->callbacks.leave(&coords, gboard->callbacks.data);
    }
    return 1;
}

/*
 * Call the user-supplied press callback. Signal handler for the
 * button-press-event.
 */
static int _um_gui_board_press(GUIBoard *gboard, GdkEvent *event) {
    if (gboard->callbacks.press) {
        gboard->callbacks.press(&gboard->active, event,
                                gboard->callbacks.data);
    }
    return 1;
}

/*
 * Call the user-supplied release callback. Signal handler for the
 * button-release-event.
 */
static int _um_gui_board_release(GUIBoard *gboard, GdkEvent *event) {
    if (gboard->active.x != -1 && gboard->active.y != -1 &&
        ((GdkEventButton *) event)->button == 1) {
        um_gui_board_toggle_tiles(gboard, &gboard->active, 0, 0, -1);
    }
    if (gboard->callbacks.release) {
        gboard->callbacks.release(&gboard->active, event,
                                    gboard->callbacks.data);
    }
    return 1;
}

/*
 * Call the user-supplied toggle callback. Signal handler for the toggled
 * event.
 */
static void _um_gui_board_toggle(GtkWidget *widget, GUIBoard *gboard) {
    Coords coords;
    _um_gui_board_tile_coords(gboard, widget, &coords);
    if (gboard->callbacks.toggle) {
        gboard->callbacks.toggle(&coords, gboard->callbacks.data);
    }
}

/*
 * Resize the given GUIBoard to the given width and height. Existing tiles will
 * be rebuilt at the next board update. New tiles are assumed blank.
 */
static int _um_gui_board_resize(GUIBoard *gboard, int width, int height) {
    int i, j;
    GtkGrid *grid = GTK_GRID(gboard->widget);
    if (_um_gui_board_state_create(gboard, width, height)) {
        return 1;
    }
    if (gboard->width < width) {
        for (i = gboard->width; i < width; ++i) {
            for (j = 0; j < gboard->height; ++j) {
                gtk_grid_attach(grid, _um_gui_board_build_tile(gboard),
                                i, j, 1, 1);
            }
        }
    } else if (gboard->width > width) {
        for (i = gboard->width - 1; i >= width; --i) {
            for (j = 0; j < gboard->height; ++j) {
                gtk_widget_destroy(gtk_grid_get_child_at(grid, i, j));
            }
        }
    }
    gboard->width = width;
    if (gboard->height < height) {
        for (j = gboard->height; j < height; ++j) {
            for (i = 0; i < gboard->width; ++i) {
                gtk_grid_attach(grid, _um_gui_board_build_tile(gboard),
                                i, j, 1, 1);
            }
        }
    } else if (gboard->height > height) {
        for (j = gboard->height - 1; j >= height; --j) {
            for (i = 0; i < gboard->width; ++i) {
                gtk_widget_destroy(gtk_grid_get_child_at(grid, i, j));
            }
        }
    }
    gboard->height = height;
    gtk_widget_show_all(gboard->widget);
    return 0;
}

/*
 * Create new internal state. If there is existing state, it will be copied
 * (and destroyed); this requires the GUIBoard's width and height match the
 * existing state.
 */
static int _um_gui_board_state_create(GUIBoard *gboard, int width,
                                        int height) {
    int *revealed,
        **revealed_pointers,
        *face,
        **face_pointers,
        i, j;
    revealed_pointers = malloc(sizeof(int *) * width);
    if (!revealed_pointers) {
        return 1;
    }
    revealed = malloc(sizeof(int) * width * height);
    if (!revealed) {
        free(revealed_pointers);
        return 1;
    }
    face_pointers = malloc(sizeof(int *) * width);
    if (!face_pointers) {
        free(revealed_pointers);
        free(revealed);
        return 1;
    }
    face = malloc(sizeof(int) * width * height);
    if (!face) {
        free(revealed_pointers);
        free(revealed);
        free(face_pointers);
        return 1;
    }
    for (i = 0; i < width; ++i) {
        revealed_pointers[i] = &revealed[i * height];
        face_pointers[i] = &face[i * height];
        for (j = 0; j < height; ++j) {
            revealed_pointers[i][j] = 0;
            face_pointers[i][j] = 0;
        }
    }
    if (gboard->revealed) {
        for (i = 0; i < gboard->width && i < width; ++i) {
            for (j = 0; j < gboard->height && j < height; ++j) {
                revealed_pointers[i][j] = gboard->revealed[i][j];
            }
        }
    }
    if (gboard->face) {
        for (i = 0; i < gboard->width && i < width; ++i) {
            for (j = 0; j < gboard->height && j < height; ++j) {
                face_pointers[i][j] = gboard->face[i][j];
            }
        }
    }
    _um_gui_board_state_destroy(gboard);
    gboard->revealed = revealed_pointers;
    gboard->face = face_pointers;
    return 0;
}

/*
 * Destroy the GUIBoard's internal state.
 */
static void _um_gui_board_state_destroy(GUIBoard *gboard) {
    if (gboard->revealed) {
        free(&gboard->revealed[0][0]);
        free(gboard->revealed);
        gboard->revealed = NULL;
    }
    if (gboard->face) {
        free(&gboard->face[0][0]);
        free(gboard->face);
        gboard->face = NULL;
    }
}

/*
 * Put the Coords of the given tile widget (the GtkStack, or one of its
 * children) in coords.
 */
static void _um_gui_board_tile_coords(const GUIBoard *gboard,
                                        GtkWidget *widget, Coords *coords) {
    if (!GTK_IS_STACK(widget)) {
        widget = gtk_widget_get_parent(widget);
    }
    gtk_container_child_get(GTK_CONTAINER(gboard->widget), widget,
                            "left-attach", &coords->x,
                            "top-attach", &coords->y,
                            NULL);
}

/*
 * Toggle the tiles in the range defined by i_start, i_end, j_start, and j_end,
 * that have the given face (face may be -1).
 */
static void _um_gui_board_toggle_range(const GUIBoard *gboard, int depressed,
                                        int face, int i_start, int i_end,
                                        int j_start, int j_end) {
    int i = i_start;
    GtkGrid *grid = GTK_GRID(gboard->widget);
    if (i < 0) {
        i = 0;
    }
    if (i_end > gboard->width) {
        i_end = gboard->width;
    }
    if (j_start < 0) {
        j_start = 0;
    }
    if (j_end > gboard->height) {
        j_end = gboard->height;
    }
    for (; i < i_end; ++i) {
        int j;
        for (j = j_start; j < j_end; ++j) {
            if (!gboard->revealed[i][j] &&
                (face < 0 || face == gboard->face[i][j])) {
                gtk_toggle_button_set_active(
                    GTK_TOGGLE_BUTTON(
                        gtk_stack_get_visible_child(
                            GTK_STACK(gtk_grid_get_child_at(grid, i, j))
                        )
                    ),
                    depressed
                );
            }
        }
    }
}

/*
 * Public functions.
 */

/*
 * Create a GUIBoard with the given callback functions and tile pixbufs.
 * face_pixbufs is an array of pixbufs, such that any face is a valid index,
 * and mine_pixbufs is a 2-array of pixbufs. See um_gui_board_update_tile for
 * details on how these are used.
 */
GUIBoard * um_gui_board_create(const GUIBoardCallbacks *callbacks,
                                GdkPixbuf **face_pixbufs,
                                GdkPixbuf **mine_pixbufs) {
    GUIBoard *gboard;
    gboard = malloc(sizeof(GUIBoard));
    if (!gboard) {
        return NULL;
    }
    gboard->active.x = -1;
    gboard->active.y = -1;
    gboard->face_pixbufs = face_pixbufs;
    gboard->mine_pixbufs = mine_pixbufs;
    gboard->widget = _um_gui_board_build(gboard);
    gboard->callbacks = *callbacks;
    gboard->revealed = NULL;
    gboard->face = NULL;
    gboard->tiles = NULL;
    gboard->width = 0;
    gboard->height = 0;
    return gboard;
}

/*
 * Destroy the GUIBoard.
 */
void um_gui_board_destroy(GUIBoard *gboard) {
    if (GTK_IS_WIDGET(gboard->widget)) {
        gtk_widget_destroy(gboard->widget);
    }
    _um_gui_board_state_destroy(gboard);
    free(gboard);
}

/*
 * Returns a pointer to the GUIBoard widget.
 */
GtkWidget * um_gui_board_get_widget(const GUIBoard *gboard) {
    return gboard->widget;
}

/*
 * Populate the GUIBoard with the given tiles. Returns 0 on success and 1 on
 * failure (in which case, the GUIBoard is destroyed).
 */
int um_gui_board_populate(GUIBoard *gboard, Tile **tiles, int width,
                            int height) {
    if ((gboard->width != width || gboard->height != height) &&
        _um_gui_board_resize(gboard, width, height)) {
        um_gui_board_destroy(gboard);
        return 1;
    }
    gboard->tiles = tiles;
    um_gui_board_update(gboard);
    return 0;
}

/*
 * Update the board widget. This should be called when the board has changed.
 * See um_gui_board_update_tile for details on how the board is updated.
 */
void um_gui_board_update(GUIBoard *gboard) {
    int i, j;
    for (i = 0; i < gboard->width; ++i) {
        for (j = 0; j < gboard->height; ++j) {
            Coords coords;
            coords.x = i;
            coords.y = j;
            um_gui_board_update_tile(gboard, &coords);
        }
    }
}

/*
 * Update the given tile. Call um_gui_board_update instead, unless certain
 * that only the given tile has changed. The widget logic is as follows:
 *
 *   - if the tile isn't revealed, the board displays a GtkToggleButton,
 *     containing a GtkImage set to the face GdkPixbuf corresponding to the
 *     tile's face property
 *
 *   - if the tile is revealed and mined, the board displays a GtkImage set
 *     to the mine GdkPixbuf corresponding to the tile's last property
 *
 *   - if the tile is revealed and not mined, the board displays a GtkLabel
 *     holding the count property
 */
void um_gui_board_update_tile(GUIBoard *gboard, const Coords *coords) {
    int x = coords->x,
        y = coords->y,
        last_revealed = gboard->revealed[x][y],
        last_face = gboard->face[x][y];
    Tile *tile = &gboard->tiles[x][y];
    if (last_revealed != tile->revealed || last_face != tile->face) {
        GtkStack *stack = GTK_STACK(
            gtk_grid_get_child_at(GTK_GRID(gboard->widget), x, y)
        );
        if (last_face != tile->face) {
            gtk_image_set_from_pixbuf(
                GTK_IMAGE(
                    gtk_bin_get_child(
                        GTK_BIN(
                            gtk_stack_get_child_by_name(stack, CHILD_BUTTON)
                        )
                    )
                ),
                gboard->face_pixbufs[tile->face]
            );
            gboard->face[x][y] = tile->face;
        }
        if (last_revealed != tile->revealed) {
            if (tile->revealed) {
                if (tile->mined) {
                    gtk_image_set_from_pixbuf(
                        GTK_IMAGE(
                            gtk_stack_get_child_by_name(stack, CHILD_IMAGE)
                        ),
                        gboard->mine_pixbufs[tile->last]
                    );
                    gtk_stack_set_visible_child_name(stack, CHILD_IMAGE);
                } else {
                    gtk_label_set_markup(
                        GTK_LABEL(
                            gtk_stack_get_child_by_name(stack, CHILD_LABEL)
                        ),
                        MARKUP_NUMBER[tile->count]
                    );
                    gtk_stack_set_visible_child_name(stack, CHILD_LABEL);
                }
            } else {
                gtk_stack_set_visible_child_name(stack, CHILD_BUTTON);
            }
            gboard->revealed[x][y] = tile->revealed;
        }
    }
}

/*
 * Toggle all tiles that have the given face (face may be -1).
 */
void um_gui_board_toggle_all(const GUIBoard *gboard, int depressed, int face) {
    _um_gui_board_toggle_range(gboard, depressed, face,
                                0, gboard->width,
                                0, gboard->height);
}

/*
 * Toggle the tiles in the square centred at centre and extending range tiles
 * in each direction (range may be 0), that have the given face (face may be
 * -1).
 */
void um_gui_board_toggle_tiles(const GUIBoard *gboard, const Coords *centre,
                                int range, int depressed, int face) {
    _um_gui_board_toggle_range(gboard, depressed, face,
                                centre->x - range, centre->x + range + 1,
                                centre->y - range, centre->y + range + 1);
}
