/*
 * unmine/gui_board.h
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef UM_GUI_BOARD_
#define UM_GUI_BOARD_

#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include "coords.h"
#include "tile.h"

/*
 * Types.
 */

typedef struct GUIBoard GUIBoard;

typedef struct {
    void (*enter)(const Coords *coords, void *data);
    void (*leave)(const Coords *coords, void *data);
    void (*press)(const Coords *coords, GdkEvent *event, void *data);
    void (*release)(const Coords *coords, GdkEvent *event, void *data);
    void (*toggle)(const Coords *coords, void *data);
    void *data;
} GUIBoardCallbacks;

/*
 * Functions.
 */

/*
 * Create a GUIBoard with the given callback functions and tile pixbufs.
 * face_pixbufs is an array of pixbufs, such that any face is a valid index,
 * and mine_pixbufs is a 2-array of pixbufs. See um_gui_board_update_tile for
 * details on how these are used.
 */
GUIBoard * um_gui_board_create(const GUIBoardCallbacks *callbacks,
                                GdkPixbuf **face_pixbufs,
                                GdkPixbuf **mine_pixbufs);

/*
 * Destroy the GUIBoard.
 */
void um_gui_board_destroy(GUIBoard *gboard);

/*
 * Returns a pointer to the GUIBoard widget.
 */
GtkWidget * um_gui_board_get_widget(const GUIBoard *gboard);

/*
 * Populate the GUIBoard with the given tiles. Returns 0 on success and 1 on
 * failure (in which case, the GUIBoard is destroyed).
 */
int um_gui_board_populate(GUIBoard *gboard, Tile **tiles, int width,
                            int height);

/*
 * Update the board widget. This should be called when the board has changed.
 * See um_gui_board_update_tile for details on how the board is updated.
 */
void um_gui_board_update(GUIBoard *gboard);

/*
 * Update the given tile. Call um_gui_board_update instead, unless certain
 * that only the given tile has changed. The widget logic is as follows:
 *
 *   - if the tile isn't revealed, the board displays a GtkToggleButton,
 *     containing a GtkImage set to the face GdkPixbuf corresponding to the
 *     tile's face property
 *
 *   - if the tile is revealed and mined, the board displays a GtkImage set
 *     to the mine GdkPixbuf corresponding to the tile's last property
 *
 *   - if the tile is revealed and not mined, the board displays a GtkLabel
 *     holding the count property
 */
void um_gui_board_update_tile(GUIBoard *gboard, const Coords *coords);

/*
 * Toggle all tiles that have the given face (face may be -1).
 */
void um_gui_board_toggle_all(const GUIBoard *gboard, int depressed, int face);

/*
 * Toggle the tiles in the square centred at centre and extending range tiles
 * in each direction (range may be 0), that have the given face (face may be
 * -1).
 */
void um_gui_board_toggle_tiles(const GUIBoard *gboard, const Coords *centre,
                                int range, int depressed, int face);

#endif /* UM_GUI_BOARD_ */
