/*
 * unmine/gui_classic.c
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "gui_classic.h"

#include <stdlib.h>

#include "classic.h"
#include "classic_config.h"
#include "coords.h"
#include "gui.h"
#include "gui_board.h"
#include "gui_params.h"
#include "tile.h"

/*
 * Time delay (milliseconds) between timer refreshes.
 */
static const int TIMER_DELAY = 250;

/*
 * Game parameter limits.
 */
static const ClassicParameters MAX_PARAMETERS = {30, 30, 30 * 30 - 2};

/*
 * Strings.
 */

static const char *MENU_C_OPTS = "Classic _Options";
static const char *MENU_C_OPTS_PARAMS = "_Mines / _Size";
static const char *MENU_C_OPTS_PARAMS_SMALL = "_Small";
static const char *MENU_C_OPTS_PARAMS_MEDIUM = "_Medium";
static const char *MENU_C_OPTS_PARAMS_LARGE = "_Large";
static const char *MENU_C_OPTS_PARAMS_CUSTOM = "_Custom...";
static const char *MENU_C_OPTS_COVER_ON_LOSS = "_Cover on Loss";
static const char *MENU_C_OPTS_NO_EXPAND_ZERO_TILES = "No E_xpand Zero Tiles";
static const char *MENU_C_OPTS_QUESTION_MARKS = "Use _Question Marks";
static const char *MENU_C_OPTS_WINMODE = "_Win Mode";
static const char *MENU_C_OPTS_WINMODE_REVEAL = "_Reveal all tiles";
static const char *MENU_C_OPTS_WINMODE_FLAGS = "_Flag all mines";
static const char *MENU_C_OPTS_WINMODE_EITHER = "_Either";
static const char *MENU_C_OPTS_WINMODE_BOTH = "_Both";
static const char * const PATH_FACE_IMAGES[] = {
    NULL,
    "icons/8x8/flag.png",
    "icons/8x8/question_mark.png",
    "icons/8x8/incorrect_flag.png"
};
static const char * const PATH_MINE_IMAGES[] = {
    "icons/8x8/mine.png",
    "icons/8x8/exploded_mine.png"
};
static const char * const PATH_STATE_IMAGES[] = {
    "icons/24x24/state_not_started.png",
    "icons/24x24/state_in_progress.png",
    "icons/24x24/state_lost.png",
    "icons/24x24/state_won.png"
};

/*
 * Types.
 */

struct GUIClassic {
    config_setting_t *config;
    GdkPixbuf *face_pixbufs[4],
              *mine_pixbufs[2];
    GtkWidget *widget_container,
              *widget_flag_count,
              *widget_menu,
              *widget_preset,
              *widget_state,
              *widget_state_images[4],
              *widget_timer;
    GUIBoard *gboard;
    GUIParams *gparams;
    int mouse_state[2];
    int timer_id;
    ClassicOptions options;
    ClassicParameters params;
    ClassicState state;
    ClassicGame *game;
    Tile **tiles;
};

/*
 * Private functions.
 */

/* Initialization. */
static void _um_gui_classic_config_load(GUIClassic *gui);
static void _um_gui_classic_load_resources(GUIClassic *gui);
static void _um_gui_classic_destroy_resources(GUIClassic *gui);
static void _um_gui_classic_build(GUIClassic *gui, GtkWidget *window);
static void _um_gui_classic_build_controls(GUIClassic *gui);
static void _um_gui_classic_build_gboard(GUIClassic *gui);
static void _um_gui_classic_build_menu(GUIClassic *gui);
static int _um_gui_classic_init_game(GUIClassic *gui);

/* UI updates. */
static int _um_gui_classic_new_board(GUIClassic *gui);
static void _um_gui_classic_update_board(Tile **tiles, void *data);
static void _um_gui_classic_update_flags(int flag_count, void *data);
static void _um_gui_classic_update_state(ClassicState state, void *data);
static void _um_gui_classic_update_tile(Tile *tile, void *data);
static int  _um_gui_classic_update_timer(void *data);

/* Game actions. */
static void _um_gui_classic_params(GUIClassic *gui);

/* GUIBoard callbacks. */
static void _um_gui_classic_mouse_enter(const Coords *coords, void *data);
static void _um_gui_classic_mouse_leave(const Coords *coords, void *data);
static void _um_gui_classic_mouse_press(const Coords *coords, GdkEvent *event,
                                        void *data);
static void _um_gui_classic_mouse_release(const Coords *coords,
                                            GdkEvent *event, void *data);

/* Signal handlers for menu items and the reset button. */
static void _um_gui_classic_params_custom(GtkWidget *menu_item,
                                            GUIClassic *gui);
static void _um_gui_classic_params_preset(GUIClassic *gui,
                                            GtkWidget *menu_item,
                                            const ClassicParameters *preset);
static void _um_gui_classic_params_small(GtkWidget *menu_item,
                                            GUIClassic *gui);
static void _um_gui_classic_params_medium(GtkWidget *menu_item,
                                            GUIClassic *gui);
static void _um_gui_classic_params_large(GtkWidget *menu_item,
                                            GUIClassic *gui);
static void _um_gui_classic_reset(GUIClassic *gui);
static void _um_gui_classic_toggle_cover_on_loss(GtkWidget *menu_item,
                                                    GUIClassic *gui);
static void _um_gui_classic_toggle_no_expand_zero_tiles(GtkWidget *menu_item,
                                                        GUIClassic *gui);
static void _um_gui_classic_toggle_question_marks(GtkWidget *menu_item,
                                                    GUIClassic *gui);
static void _um_gui_classic_win_mode(GUIClassic *gui, GtkWidget *menu_item,
                                        const ClassicWinMode win_mode);
static void _um_gui_classic_win_mode_reveal(GtkWidget *menu_item,
                                            GUIClassic *gui);
static void _um_gui_classic_win_mode_flags(GtkWidget *menu_item,
                                            GUIClassic *gui);
static void _um_gui_classic_win_mode_either(GtkWidget *menu_item,
                                            GUIClassic *gui);
static void _um_gui_classic_win_mode_both(GtkWidget *menu_item,
                                            GUIClassic *gui);

/*
 * Default values.
 */
static const ClassicCallbacks CLASSIC_CALLBACKS = {
    _um_gui_classic_update_board,
    _um_gui_classic_update_flags,
    _um_gui_classic_update_state,
    _um_gui_classic_update_tile,
    NULL
};
static const ClassicOptions CLASSIC_OPTIONS = {0, 0, 0, EITHER};
static const GUIBoardCallbacks GBOARD_CALLBACKS = {
    _um_gui_classic_mouse_enter,
    _um_gui_classic_mouse_leave,
    _um_gui_classic_mouse_press,
    _um_gui_classic_mouse_release,
    NULL,
    NULL
};

/*
 * Load configuration.
 */
static void _um_gui_classic_config_load(GUIClassic *gui) {
    gui->options = CLASSIC_OPTIONS;
    gui->params = UM_CLASSIC_DEFAULT;
    um_classic_config_load(gui->config, &gui->options, &gui->params);
}

/*
 * Load images and pixbufs. These are managed separately because, unlike other
 * widgets, these aren't implictly destroyed with their parents; they must be
 * explicitly destroyed with _um_gui_classic_destroy_resources.
 */
static void _um_gui_classic_load_resources(GUIClassic *gui) {
    int i;
    for (i = BLANK; i < 4; ++i) {
        if (PATH_FACE_IMAGES[i]) {
            gui->face_pixbufs[i] = gdk_pixbuf_new_from_file(
                PATH_FACE_IMAGES[i], NULL
            );
        } else {
            gui->face_pixbufs[i] = NULL;
        }
    }
    for (i = 0; i < 2; ++i) {
        gui->mine_pixbufs[i] = gdk_pixbuf_new_from_file(
            PATH_MINE_IMAGES[i], NULL
        );
    }
    for (i = NOT_STARTED; i < 4; ++i) {
        gui->widget_state_images[i] = gtk_image_new_from_file(
            PATH_STATE_IMAGES[i]
        );
        g_object_ref(gui->widget_state_images[i]);
    }
}

/*
 * Destroy images and pixbufs.
 */
static void _um_gui_classic_destroy_resources(GUIClassic *gui) {
    int i;
    for (i = BLANK; i < 4; ++i) {
        if (gui->face_pixbufs[i]) {
            g_object_unref(gui->face_pixbufs[i]);
        }
    }
    for (i = 0; i < 2; ++i) {
        g_object_unref(gui->mine_pixbufs[i]);
    }
    for (i = NOT_STARTED; i < 4; i += 1) {
        g_object_unref(gui->widget_state_images[i]);
    }
}

/*
 * Build the GUI (creates the top-level container and calls the other build
 * functions).
 */
static void _um_gui_classic_build(GUIClassic *gui, GtkWidget *window) {
    gui->widget_container = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);
    _um_gui_classic_build_menu(gui);
    _um_gui_classic_build_controls(gui);
    _um_gui_classic_build_gboard(gui);
    gui->gparams = um_gui_params_create(
        window,
        UM_CLASSIC_MIN.width, UM_CLASSIC_MIN.height, UM_CLASSIC_MIN.num_mines,
        MAX_PARAMETERS.width, MAX_PARAMETERS.height, MAX_PARAMETERS.num_mines
    );
}

/*
 * Build the controls (mine flag count, state display / reset button, and
 * timer) and pack them into the top-level container.
 */
static void _um_gui_classic_build_controls(GUIClassic *gui) {
    GtkWidget *controls = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 0);
    gui->widget_flag_count = gtk_label_new(NULL);
    gui->widget_state = gtk_button_new();
    g_signal_connect_swapped(gui->widget_state, "clicked",
                                G_CALLBACK(_um_gui_classic_reset), gui);
    gui->widget_timer = gtk_label_new(NULL);
    gtk_box_pack_start(GTK_BOX(controls), gui->widget_flag_count, 0, 0, 8);
    gtk_box_set_center_widget(GTK_BOX(controls), gui->widget_state);
    gtk_box_pack_end(GTK_BOX(controls), gui->widget_timer, 0, 0, 8);
    gtk_box_pack_start(GTK_BOX(gui->widget_container), controls, 0, 0, 8);
}

/*
 * Build the GUI Board and pack it into the top-level container.
 */
static void _um_gui_classic_build_gboard(GUIClassic *gui) {
    GUIBoardCallbacks callbacks = GBOARD_CALLBACKS;
    callbacks.data = gui;
    gui->mouse_state[0] = 0;
    gui->mouse_state[1] = 0;
    gui->gboard = um_gui_board_create(&callbacks, gui->face_pixbufs,
                                        gui->mine_pixbufs);
    gtk_box_pack_start(GTK_BOX(gui->widget_container),
                        um_gui_board_get_widget(gui->gboard), 1, 0, 0);
}

/*
 * Build the game menu.
 */
static void _um_gui_classic_build_menu(GUIClassic *gui) {
    GSList *params_group = NULL,
           *win_mode_group = NULL;
    GtkWidget *both,
              *c_opts,
              *c_opts_menu,
              *cover_on_loss,
              *custom,
              *either,
              *flags,
              *large,
              *medium,
              *no_expand_zero_tiles,
              *params,
              *params_menu,
              *question_marks,
              *reveal,
              *sep_c_opts,
              *sep_params,
              *small,
              *win_mode,
              *win_mode_active,
              *win_mode_menu;

    /* Create menu items. */
    c_opts = gtk_menu_item_new_with_mnemonic(MENU_C_OPTS);
    params = gtk_menu_item_new_with_mnemonic(MENU_C_OPTS_PARAMS);
    small = gtk_radio_menu_item_new_with_mnemonic(params_group,
                                                    MENU_C_OPTS_PARAMS_SMALL);
    params_group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(small));
    medium = gtk_radio_menu_item_new_with_mnemonic(params_group,
                                                    MENU_C_OPTS_PARAMS_MEDIUM);
    large = gtk_radio_menu_item_new_with_mnemonic(params_group,
                                                    MENU_C_OPTS_PARAMS_LARGE);
    sep_params = gtk_separator_menu_item_new();
    custom = gtk_radio_menu_item_new_with_mnemonic(params_group,
                                                    MENU_C_OPTS_PARAMS_CUSTOM);
    sep_c_opts = gtk_separator_menu_item_new();
    cover_on_loss = gtk_check_menu_item_new_with_mnemonic(
                                                    MENU_C_OPTS_COVER_ON_LOSS);
    no_expand_zero_tiles = gtk_check_menu_item_new_with_mnemonic(
                                            MENU_C_OPTS_NO_EXPAND_ZERO_TILES);
    question_marks = gtk_check_menu_item_new_with_mnemonic(
                                                MENU_C_OPTS_QUESTION_MARKS);
    win_mode = gtk_menu_item_new_with_mnemonic(MENU_C_OPTS_WINMODE);
    reveal = gtk_radio_menu_item_new_with_mnemonic(win_mode_group,
                                                MENU_C_OPTS_WINMODE_REVEAL);
    win_mode_group = gtk_radio_menu_item_get_group(
                                                GTK_RADIO_MENU_ITEM(reveal));
    flags = gtk_radio_menu_item_new_with_mnemonic(win_mode_group,
                                                    MENU_C_OPTS_WINMODE_FLAGS);
    either = gtk_radio_menu_item_new_with_mnemonic(win_mode_group,
                                                MENU_C_OPTS_WINMODE_EITHER);
    both = gtk_radio_menu_item_new_with_mnemonic(win_mode_group,
                                                    MENU_C_OPTS_WINMODE_BOTH);

    /* Set current option values. */
    if (gui->params.width == UM_CLASSIC_SMALL.width &&
        gui->params.height == UM_CLASSIC_SMALL.height &&
        gui->params.num_mines == UM_CLASSIC_SMALL.num_mines) {
        gui->widget_preset = small;
    } else if (gui->params.width == UM_CLASSIC_MEDIUM.width &&
               gui->params.height == UM_CLASSIC_MEDIUM.height &&
               gui->params.num_mines == UM_CLASSIC_MEDIUM.num_mines) {
        gui->widget_preset = medium;
    } else if (gui->params.width == UM_CLASSIC_LARGE.width &&
               gui->params.height == UM_CLASSIC_LARGE.height &&
               gui->params.num_mines == UM_CLASSIC_LARGE.num_mines) {
        gui->widget_preset = large;
    } else {
        gui->widget_preset = custom;
    }
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(gui->widget_preset),
                                    TRUE);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(cover_on_loss),
                                    gui->options.cover_on_loss);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(no_expand_zero_tiles),
                                    gui->options.no_expand_zero_tiles);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(question_marks),
                                    gui->options.question_marks);
    if (gui->options.win_mode == REVEAL) {
        win_mode_active = reveal;
    } else if (gui->options.win_mode == FLAGS) {
        win_mode_active = flags;
    } else if (gui->options.win_mode == BOTH) {
        win_mode_active = both;
    } else {
        win_mode_active = either;
    }
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(win_mode_active), TRUE);

    /* Create menus and fill them with menu items. */
    params_menu = gtk_menu_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(params_menu), small);
    gtk_menu_shell_append(GTK_MENU_SHELL(params_menu), medium);
    gtk_menu_shell_append(GTK_MENU_SHELL(params_menu), large);
    gtk_menu_shell_append(GTK_MENU_SHELL(params_menu), sep_params);
    gtk_menu_shell_append(GTK_MENU_SHELL(params_menu), custom);
    win_mode_menu = gtk_menu_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(win_mode_menu), reveal);
    gtk_menu_shell_append(GTK_MENU_SHELL(win_mode_menu), flags);
    gtk_menu_shell_append(GTK_MENU_SHELL(win_mode_menu), either);
    gtk_menu_shell_append(GTK_MENU_SHELL(win_mode_menu), both);
    c_opts_menu = gtk_menu_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(c_opts_menu), params);
    gtk_menu_shell_append(GTK_MENU_SHELL(c_opts_menu), sep_c_opts);
    gtk_menu_shell_append(GTK_MENU_SHELL(c_opts_menu), cover_on_loss);
    gtk_menu_shell_append(GTK_MENU_SHELL(c_opts_menu), no_expand_zero_tiles);
    gtk_menu_shell_append(GTK_MENU_SHELL(c_opts_menu), question_marks);
    gtk_menu_shell_append(GTK_MENU_SHELL(c_opts_menu), win_mode);

    /* Link menu items to submenus. */
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(params), params_menu);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(win_mode), win_mode_menu);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(c_opts), c_opts_menu);

    /* Connect signal handlers. */
    g_signal_connect(small, "activate",
                        G_CALLBACK(_um_gui_classic_params_small), gui);
    g_signal_connect(medium, "activate",
                        G_CALLBACK(_um_gui_classic_params_medium), gui);
    g_signal_connect(large, "activate",
                        G_CALLBACK(_um_gui_classic_params_large), gui);
    g_signal_connect(custom, "activate",
                        G_CALLBACK(_um_gui_classic_params_custom), gui);
    g_signal_connect(cover_on_loss, "toggled",
                        G_CALLBACK(_um_gui_classic_toggle_cover_on_loss), gui);
    g_signal_connect(no_expand_zero_tiles, "toggled",
                G_CALLBACK(_um_gui_classic_toggle_no_expand_zero_tiles), gui);
    g_signal_connect(question_marks, "toggled",
                    G_CALLBACK(_um_gui_classic_toggle_question_marks), gui);
    g_signal_connect(reveal, "toggled",
                        G_CALLBACK(_um_gui_classic_win_mode_reveal), gui);
    g_signal_connect(flags, "toggled",
                        G_CALLBACK(_um_gui_classic_win_mode_flags), gui);
    g_signal_connect(either, "toggled",
                        G_CALLBACK(_um_gui_classic_win_mode_either), gui);
    g_signal_connect(both, "toggled",
                        G_CALLBACK(_um_gui_classic_win_mode_both), gui);

    gui->widget_menu = c_opts;
}

/*
 * Initialize the game.
 */
static int _um_gui_classic_init_game(GUIClassic *gui) {
    ClassicCallbacks callbacks = CLASSIC_CALLBACKS;
    callbacks.data = gui;
    gui->state = NOT_STARTED;
    gui->game = um_classic_create(&callbacks, &gui->options, &gui->params);
    if (!gui->game) {
        return 1;
    }
    gui->tiles = um_classic_tiles(gui->game);
    if (_um_gui_classic_new_board(gui)) {
        um_classic_destroy(gui->game);
        return 1;
    }
    return 0;
}

/*
 * Update the UI after a board change. Populating the GUIBoard can fail, in
 * which case this returns 1.
 */
static int _um_gui_classic_new_board(GUIClassic *gui) {
    if (um_gui_board_populate(gui->gboard, gui->tiles, gui->params.width,
                                gui->params.height)) {
        return 1;
    }
    um_classic_update(gui->game);
    um_gui_shrink();
    _um_gui_classic_update_timer(gui);
    return 0;
}

/*
 * Update the GUI Board. This is a ClassicGame callback function.
 */
static void _um_gui_classic_update_board(Tile **tiles, void *data) {
    GUIClassic *gui = (GUIClassic *) data;
    um_gui_board_update(gui->gboard);
}

/*
 * Set the mine flag count display to the current flag count. This is a
 * ClassicGame callback function.
 */
static void _um_gui_classic_update_flags(int flag_count, void *data) {
    GUIClassic *gui = (GUIClassic *) data;
    char buffer[] = ".....";
    sprintf(buffer, "%d", flag_count);
    gtk_label_set_text(GTK_LABEL(gui->widget_flag_count), buffer);
}

/*
 * Transition between states. This is a ClassicGame callback function.
 */
static void _um_gui_classic_update_state(ClassicState state, void *data) {
    GUIClassic *gui = (GUIClassic *) data;
    if (state != gui->state) {
        if (state == IN_PROGRESS) {
            gui->timer_id = g_timeout_add(TIMER_DELAY,
                                        &_um_gui_classic_update_timer, gui);
        } else {
            if (state == WON) {
                /* TODO: save score. */
            }
            if (gui->timer_id) {
                g_source_remove(gui->timer_id);
                gui->timer_id = 0;
            }
        }
        gui->state = state;
    }
    gtk_button_set_image(GTK_BUTTON(gui->widget_state),
                            gui->widget_state_images[state]);
}

/*
 * Update the given tile on the GUI Board. This is a ClassicGame callback
 * function.
 */
static void _um_gui_classic_update_tile(Tile *tile, void *data) {
    GUIClassic *gui = (GUIClassic *) data;
    Coords coords;
    coords.x = tile->x;
    coords.y = tile->y;
    um_gui_board_update_tile(gui->gboard, &coords);
}

/*
 * Set the game time display to the current game time. This is a g_timeout
 * function; it's called repeatedly until it returns G_SOURCE_REMOVE.
 */
static int _um_gui_classic_update_timer(void *data) {
    ClassicState tstate;
    char buffer[] = "...";
    double seconds;
    GUIClassic *gui = (GUIClassic *) data;
    if (gui->widget_timer == NULL) {
        return G_SOURCE_REMOVE;
    }
    seconds = um_classic_time(gui->game);
    if (seconds < 1000) {
        sprintf(buffer, "%3.f", seconds);
    }
    gtk_label_set_text(GTK_LABEL(gui->widget_timer), buffer);
    tstate = um_classic_state(gui->game);
    if (tstate != IN_PROGRESS) {
        gui->timer_id = 0;
        return G_SOURCE_REMOVE;
    }
    return G_SOURCE_CONTINUE;
}

/*
 * Set the game parameters. This can fail, in which case the application will
 * be terminated.
 */
static void _um_gui_classic_params(GUIClassic *gui) {
    gui->tiles = um_classic_parameters(gui->game, &gui->params);
    if (!gui->tiles || _um_gui_classic_new_board(gui)) {
        gtk_main_quit();
    }
}

/*
 * Depress the surrounding tiles if both mouse buttons are down. This is a
 * GUIBoard callback function.
 */
static void _um_gui_classic_mouse_enter(const Coords *coords, void *data) {
    GUIClassic *gui = (GUIClassic *) data;
    if (gui->mouse_state[0] && gui->mouse_state[1]) {
        um_gui_board_toggle_tiles(gui->gboard, coords, 1, 1, BLANK);
        um_gui_board_toggle_tiles(gui->gboard, coords, 1, 1, QUESTION_MARK);
    }
}

/*
 * Return surrounding tiles to their normal state if the left mouse button is
 * down. This is a GUIBoard callback function.
 */
static void _um_gui_classic_mouse_leave(const Coords *coords, void *data) {
    GUIClassic *gui = (GUIClassic *) data;
    if (gui->mouse_state[0]) {
        um_gui_board_toggle_tiles(gui->gboard, coords, 1, 0, BLANK);
        um_gui_board_toggle_tiles(gui->gboard, coords, 1, 0, QUESTION_MARK);
    }
}

/*
 * Update the click state and depress the surrounding tiles if both mouse
 * buttons are down. This is a GUIBoard callback function.
 */
static void _um_gui_classic_mouse_press(const Coords *coords, GdkEvent *event,
                                        void *data) {
    GUIClassic *gui = (GUIClassic *) data;
    GdkEventButton *real_event = (GdkEventButton *) event;
    if (real_event->button == 1) {
        gui->mouse_state[0] = 1;
    } else if (real_event->button == 2) {
        gui->mouse_state[0] = 1;
        gui->mouse_state[1] = 1;
    } else if (real_event->button == 3) {
        gui->mouse_state[1] = 1;
    }
    if (gui->mouse_state[0] && gui->mouse_state[1]) {
        um_gui_board_toggle_tiles(gui->gboard, coords, 1, 1, BLANK);
        um_gui_board_toggle_tiles(gui->gboard, coords, 1, 1, QUESTION_MARK);
    }
}

/*
 * Update the click state and, depending on the click state and the button
 * pressed, do one of: reveal, chord, flag, question mark, depress tiles
 * surrounding the active tile, or nothing. This is a GUIBoard callback
 * function.
 */
static void _um_gui_classic_mouse_release(const Coords *coords,
                                            GdkEvent *event, void *data) {
    GUIClassic *gui = (GUIClassic *) data;
    ClassicState state = um_classic_state(gui->game);
    GdkEventButton *real_event = (GdkEventButton *) event;
    int x = coords->x,
        y = coords->y;
    if (real_event->button == 1) {
        gui->mouse_state[0] = 0;
    } else if (real_event->button == 2) {
        gui->mouse_state[0] = 0;
        gui->mouse_state[1] = 0;
    } else if (real_event->button == 3) {
        gui->mouse_state[1] = 0;
    }
    if (x == -1 || y == -1) {
        return;
    }
    um_gui_board_toggle_tiles(gui->gboard, coords, 1, 0, BLANK);
    um_gui_board_toggle_tiles(gui->gboard, coords, 1, 0, QUESTION_MARK);
    if (state == LOST || state == WON) {
        return;
    }
    if (real_event->button == 1) {
        if (gui->mouse_state[1]) {
            if (gui->tiles[x][y].revealed) {
                um_classic_reveal_around(gui->game, coords);
            }
        } else if (gui->tiles[x][y].face == BLANK) {
            um_classic_reveal(gui->game, coords);
        }
    } else if (real_event->button == 2) {
        if (gui->tiles[x][y].revealed) {
            um_classic_reveal_around(gui->game, coords);
        }
    } else if (real_event->button == 3) {
        if (!gui->mouse_state[0] && !gui->tiles[x][y].revealed) {
            if (gui->tiles[x][y].face == BLANK) {
                um_classic_flag(gui->game, coords);
            } else if (gui->tiles[x][y].face == MINE_FLAG) {
                if (gui->options.question_marks) {
                    um_classic_flag(gui->game, coords);
                    um_classic_question(gui->game, coords);
                } else {
                    um_classic_flag(gui->game, coords);
                }
            } else if (gui->tiles[x][y].face == QUESTION_MARK) {
                um_classic_question(gui->game, coords);
            }
        }
    }
}

/*
 * Run the game parameters dialog. If canceled, reset the form; if affirmed,
 * set the new game parameters. Signal handler for the 'Custom' menu item.
 */
static void _um_gui_classic_params_custom(GtkWidget *menu_item,
                                            GUIClassic *gui) {
    if (!gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menu_item))) {
        return;
    }
    if (um_gui_params_prompt(gui->gparams, &gui->params.width,
                                &gui->params.height, &gui->params.num_mines)) {
        _um_gui_classic_params(gui);
        gui->widget_preset = menu_item;
    } else {
        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(gui->widget_preset),
                                        TRUE);
    }
}

/*
 * Set the game parameters to the given preset.
 */
static void _um_gui_classic_params_preset(GUIClassic *gui,
                                            GtkWidget *menu_item,
                                            const ClassicParameters *preset) {
    if (gui->widget_preset != menu_item &&
        gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menu_item))) {
        gui->params = *preset;
        _um_gui_classic_params(gui);
        gui->widget_preset = menu_item;
    }
}

/*
 * Set the game parameters to small. Signal handler for the small menu item.
 */
static void _um_gui_classic_params_small(GtkWidget *menu_item,
                                            GUIClassic *gui) {
    _um_gui_classic_params_preset(gui, menu_item, &UM_CLASSIC_SMALL);
}

/*
 * Set the game parameters to medium. Signal handler for the medium menu item.
 */
static void _um_gui_classic_params_medium(GtkWidget *menu_item,
                                            GUIClassic *gui) {
    _um_gui_classic_params_preset(gui, menu_item, &UM_CLASSIC_MEDIUM);
}

/*
 * Set the game parameters to large. Signal handler for the large menu item.
 */
static void _um_gui_classic_params_large(GtkWidget *menu_item,
                                            GUIClassic *gui) {
    _um_gui_classic_params_preset(gui, menu_item, &UM_CLASSIC_LARGE);
}

/*
 * Reset the game. Signal handler for the reset button.
 */
static void _um_gui_classic_reset(GUIClassic *gui) {
    if (um_classic_state(gui->game) != NOT_STARTED) {
        um_classic_reset(gui->game);
        _um_gui_classic_update_timer(gui);
    }
}

/*
 * Set the cover on loss option. Signal handler for the 'Cover on Loss' menu
 * item.
 */
static void _um_gui_classic_toggle_cover_on_loss(GtkWidget *menu_item,
                                                    GUIClassic *gui) {
    gui->options.cover_on_loss = gtk_check_menu_item_get_active(
        GTK_CHECK_MENU_ITEM(menu_item)
    );
    um_classic_options(gui->game, &gui->options);
}

/*
 * Set the no expand zero tiles option. Signal handler for the 'No Expand Zero
 * Tiles' menu item.
 */
static void _um_gui_classic_toggle_no_expand_zero_tiles(GtkWidget *menu_item,
                                                        GUIClassic *gui) {
    gui->options.no_expand_zero_tiles = gtk_check_menu_item_get_active(
        GTK_CHECK_MENU_ITEM(menu_item)
    );
    um_classic_options(gui->game, &gui->options);
}

/*
 * Set the question marks option. Signal handler for the 'Use Question Marks'
 * menu item.
 */
static void _um_gui_classic_toggle_question_marks(GtkWidget *menu_item,
                                                    GUIClassic *gui) {
    gui->options.question_marks = gtk_check_menu_item_get_active(
        GTK_CHECK_MENU_ITEM(menu_item)
    );
    um_classic_options(gui->game, &gui->options);
}

/*
 * If checked, set the Win Mode. Signal handler for the options in the
 * 'Win Mode' submenu.
 */
static void _um_gui_classic_win_mode(GUIClassic *gui, GtkWidget *menu_item,
                                        const ClassicWinMode win_mode) {
    if (gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menu_item))) {
        gui->options.win_mode = win_mode;
        um_classic_options(gui->game, &gui->options);
    }
}

/*
 * Set the Win Mode to reveal. Signal handler for the reveal menu item.
 */
static void _um_gui_classic_win_mode_reveal(GtkWidget *menu_item,
                                            GUIClassic *gui) {
    _um_gui_classic_win_mode(gui, menu_item, REVEAL);
}

/*
 * Set the Win Mode to flags. Signal handler for the flags menu item.
 */
static void _um_gui_classic_win_mode_flags(GtkWidget *menu_item,
                                            GUIClassic *gui) {
    _um_gui_classic_win_mode(gui, menu_item, FLAGS);
}

/*
 * Set the Win Mode to either. Signal handler for the either menu item.
 */
static void _um_gui_classic_win_mode_either(GtkWidget *menu_item,
                                            GUIClassic *gui) {
    _um_gui_classic_win_mode(gui, menu_item, EITHER);
}

/*
 * Set the Win Mode to both. Signal handler for the both menu item.
 */
static void _um_gui_classic_win_mode_both(GtkWidget *menu_item,
                                            GUIClassic *gui) {
    _um_gui_classic_win_mode(gui, menu_item, BOTH);
}

/*
 * Public functions.
 */

/*
 * Initialize the interface.
 */
GUIClassic * um_gui_classic_init(GtkWidget *window, config_setting_t *config) {
    GUIClassic *gui;
    gui = malloc(sizeof(GUIClassic));
    if (!gui) {
        return NULL;
    }
    gui->config = config;
    _um_gui_classic_config_load(gui);
    _um_gui_classic_load_resources(gui);
    _um_gui_classic_build(gui, window);
    if (_um_gui_classic_init_game(gui)) {
        _um_gui_classic_destroy_resources(gui);
        free(gui);
        return NULL;
    }
    return gui;
}

/*
 * Destroy the interface.
 */
void um_gui_classic_destroy(GUIClassic *gui) {
    um_classic_config_save(gui->config, &gui->options, &gui->params);
    um_classic_destroy(gui->game);
    um_gui_params_destroy(gui->gparams);
    um_gui_board_destroy(gui->gboard);
    _um_gui_classic_destroy_resources(gui);
    free(gui);
}

/*
 * Get the 'Classic Options' menu widget (GtkMenuItem).
 */
GtkWidget * um_gui_classic_get_menu(GUIClassic *gui) {
    return gui->widget_menu;
}

/*
 * Get the top-level container widget (GtkBox).
 */
GtkWidget * um_gui_classic_get_ui(GUIClassic *gui) {
    return gui->widget_container;
}
