/*
 * unmine/gui_flags.c
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "gui_flags.h"

#include <stdlib.h>

#include "coords.h"
#include "flags.h"
#include "flags_ai.h"
#include "flags_config.h"
#include "gui.h"
#include "gui_board.h"
#include "gui_params.h"
#include "tile.h"

/*
 * Constants.
 */
#define _UM_GUI_FLAGS_MAX_PLAYERS 7

/*
 * Game parameter limits.
 */
static const FlagsParameters MAX_PARAMETERS = {30, 30, 30 * 30 - 2, 0, 0, 0};

/*
 * Strings.
 */

static const char *DIALOG_CANCEL = "Cancel";
static const char *DIALOG_OK = "OK";
static const char *DIALOG_BOMB_COUNT = "Bomb _Count";
static const char *DIALOG_BOMB_RANGE = "Bomb _Range";
static const char *DIALOG_BOMB_TITLE = "Set Bomb Count / Range";
static const char *DIALOG_PLAYERS_HOTSEAT = "Hotseat";
static const char *DIALOG_PLAYERS_NONE = "None";
static const char *DIALOG_PLAYERS_TITLE = "Set Players";
static const char *MENU = "Flags _Options";
static const char *MENU_PARAMS = "_Mines / _Size";
static const char *MENU_PARAMS_STANDARD = "_Standard";
static const char *MENU_PARAMS_CUSTOM = "_Custom...";
static const char *MENU_PLAYERS = "_Players...";
static const char *MENU_ALLOW_TIES = "_Allow Ties";
static const char *MENU_BOMB_COUNT_RANGE = "_Bomb Count / Range...";
static const char *MENU_BOMB_ENDS_TURN = "Bomb _Ends Turn";
static const char *MENU_COVER_ON_DONE = "_Cover on Done";
static const char *MENU_NO_EXPAND_ZERO_TILES = "No E_xpand Zero Tiles";
static const char *PATH_BOMB_IMAGE = "icons/24x24/bomb.png";
static const char * const PATH_FACE_IMAGES[] = {
    NULL,
    "icons/8x8/flag-1.png",
    "icons/8x8/flag-2.png",
    "icons/8x8/flag-3.png",
    "icons/8x8/flag-4.png",
    "icons/8x8/flag-5.png",
    "icons/8x8/flag-6.png",
    "icons/8x8/flag-7.png",
};
static const char * const PATH_MINE_IMAGES[] = {
    "icons/8x8/mine.png",
    "icons/8x8/mine.png"
};
static const char * const PATH_PLAYER_IMAGES[] = {
    "icons/16x16/flag-1.png",
    "icons/16x16/flag-2.png",
    "icons/16x16/flag-3.png",
    "icons/16x16/flag-4.png",
    "icons/16x16/flag-5.png",
    "icons/16x16/flag-6.png",
    "icons/16x16/flag-7.png",
};
static const char *PATH_PROMPT_IMAGE = "icons/16x16/prompt.png";
static const char * const PATH_STATE_IMAGES[] = {
    "icons/24x24/state_not_started.png",
    "icons/24x24/state_in_progress.png",
    "icons/24x24/state_won.png"
};
static const char *PATH_WINNER_IMAGE = "icons/16x16/winner.png";
static const char *PLAYERS_BOMBS = "Bombs";
static const char * const PLAYERS_LABELS_AI[] = {
    "AI #1",
    "AI #2",
    "AI #3",
    "AI #4",
    "AI #5",
    "AI #6",
    "AI #7"
};
static const char * const PLAYERS_LABELS_HOTSEAT[] = {
    "Hotseat #1",
    "Hotseat #2",
    "Hotseat #3",
    "Hotseat #4",
    "Hotseat #5",
    "Hotseat #6",
    "Hotseat #7"
};
static const char *PLAYERS_SCORE = "Score";

/*
 * Types.
 */

struct GUIFlags {
    config_setting_t *config;
    GdkPixbuf *face_pixbufs[PLAYER_FLAG + _UM_GUI_FLAGS_MAX_PLAYERS],
              *mine_pixbufs[2];
    GtkWidget *widget_bomb_count,
              *widget_bomb_dialog,
              *widget_bomb_range,
              *widget_bomb_select,
              *widget_container,
              *widget_menu,
              *widget_mine_count,
              *widget_player_bombs[_UM_GUI_FLAGS_MAX_PLAYERS],
              *widget_player_icons[_UM_GUI_FLAGS_MAX_PLAYERS],
              *widget_player_labels[_UM_GUI_FLAGS_MAX_PLAYERS],
              *widget_player_prompts[_UM_GUI_FLAGS_MAX_PLAYERS],
              *widget_player_scores[_UM_GUI_FLAGS_MAX_PLAYERS],
              *widget_player_selects[_UM_GUI_FLAGS_MAX_PLAYERS],
              *widget_player_wins[_UM_GUI_FLAGS_MAX_PLAYERS],
              *widget_players_dialog,
              *widget_preset,
              *widget_state,
              *widget_state_images[NUM_FLAGS_STATES];
    GUIBoard *gboard;
    GUIParams *gparams;
    int repopulate_gboard,
        turn_player;
    FlagsAILevel ai_levels[_UM_GUI_FLAGS_MAX_PLAYERS];
    FlagsOptions options;
    FlagsParameters params;
    const FlagsPlayer *players;
    FlagsState state;
    FlagsGame *game;
};

typedef struct {
    GUIFlags *gui;
    Tile **tiles;
} BoardUpdate;

typedef struct {
    GtkWidget *widget_mine_count;
    int mine_count;
} MineCountUpdate;

typedef struct {
    GUIFlags *gui;
    const FlagsPlayer *players;
    int count;
} PlayersUpdate;

typedef struct {
    GUIFlags *gui;
    FlagsState state;
} StateUpdate;

typedef struct {
    GUIBoard *gboard;
    Coords coords;
} TileUpdate;

typedef struct {
    GUIFlags *gui;
    int turn_player;
} TurnUpdate;

/*
 * Private functions.
 */

/* Initialization. */
static void _um_gui_flags_config_load(GUIFlags *gui);
static void _um_gui_flags_load_resources(GUIFlags *gui);
static void _um_gui_flags_destroy_resources(GUIFlags *gui);
static void _um_gui_flags_build(GUIFlags *gui, GtkWidget *window);
static void _um_gui_flags_build_bomb_opts(GUIFlags *gui, GtkWidget *window);
static void _um_gui_flags_build_controls(GUIFlags *gui, GtkWidget *container);
static void _um_gui_flags_build_gboard(GUIFlags *gui, GtkWidget *container);
static void _um_gui_flags_build_menu(GUIFlags *gui);
static void _um_gui_flags_build_player_opts(GUIFlags *gui, GtkWidget *window);
static void _um_gui_flags_build_players(GUIFlags *gui);
static int  _um_gui_flags_init_game(GUIFlags *gui);

/* UI updates. */
static int  _um_gui_flags_do_board(void *update);
static int  _um_gui_flags_do_mine_count(void *update);
static int  _um_gui_flags_do_players(void *update);
static int  _um_gui_flags_do_state(void *update);
static int  _um_gui_flags_do_tile(void *update);
static int  _um_gui_flags_do_turn(void *update);

/* Game callbacks. */
static void _um_gui_flags_update_board(Tile **tiles, void *data);
static void _um_gui_flags_update_mine_count(int mine_count, void *data);
static void _um_gui_flags_update_players(const FlagsPlayer *players, int count,
                                            void *data);
static void _um_gui_flags_update_state(FlagsState state, void *data);
static void _um_gui_flags_update_tile(const Tile *tile, void *data);
static void _um_gui_flags_update_turn(int player, void *data);

/* Game actions. */
static void _um_gui_flags_params(GUIFlags *gui);

/* GUIBoard callbacks. */
static void _um_gui_flags_mouse_enter(const Coords *coords, void *data);
static void _um_gui_flags_mouse_leave(const Coords *coords, void *data);
static void _um_gui_flags_mouse_release(const Coords *coords, GdkEvent *event,
                                        void *data);

/* Signal handlers for menu items and the reset button. */
static void _um_gui_flags_bomb_options(GUIFlags *gui);
static void _um_gui_flags_params_custom(GtkWidget *menu_item, GUIFlags *gui);
static void _um_gui_flags_params_standard(GtkWidget *menu_item, GUIFlags *gui);
static void _um_gui_flags_players(GUIFlags *gui);
static void _um_gui_flags_reset(GUIFlags *gui);
static void _um_gui_flags_toggle_allow_ties(GtkWidget *menu_item,
                                            GUIFlags *gui);
static void _um_gui_flags_toggle_bomb_select(GUIFlags *gui);
static void _um_gui_flags_toggle_bomb_ends_turn(GtkWidget *menu_item,
                                                GUIFlags *gui);
static void _um_gui_flags_toggle_cover_on_done(GtkWidget *menu_item,
                                                GUIFlags *gui);
static void _um_gui_flags_toggle_no_expand_zero_tiles(GtkWidget *menu_item,
                                                        GUIFlags *gui);

/*
 * Default values.
 */

static const FlagsCallbacks FLAGS_CALLBACKS = {
    _um_gui_flags_update_board,
    _um_gui_flags_update_mine_count,
    _um_gui_flags_update_players,
    _um_gui_flags_update_state,
    _um_gui_flags_update_tile,
    _um_gui_flags_update_turn,
    NULL
};
static const FlagsOptions FLAGS_OPTIONS = {0, 0, 2, 0, 0, 1};
static const GUIBoardCallbacks GBOARD_CALLBACKS = {
    _um_gui_flags_mouse_enter,
    _um_gui_flags_mouse_leave,
    NULL,
    _um_gui_flags_mouse_release,
    NULL,
    NULL
};


/*
 * Load configuration.
 */
static void _um_gui_flags_config_load(GUIFlags *gui) {
    int i;
    for (i = 0; i < _UM_GUI_FLAGS_MAX_PLAYERS; ++i) {
        gui->ai_levels[i] = UM_FLAGS_AI_DEFAULT;
    }
    gui->options = FLAGS_OPTIONS;
    gui->params = UM_FLAGS_DEFAULT;
    gui->params.ai_levels = gui->ai_levels;
    um_flags_config_load(gui->config, &gui->options, &gui->params,
                            _UM_GUI_FLAGS_MAX_PLAYERS);
}

/*
 * Load images and pixbufs. These are managed separately because, unlike other
 * widgets, these aren't implictly destroyed with their parents; they must be
 * explicitly destroyed with _um_gui_flags_destroy_resources.
 */
static void _um_gui_flags_load_resources(GUIFlags *gui) {
    int i;
    for (i = BLANK; i < PLAYER_FLAG + _UM_GUI_FLAGS_MAX_PLAYERS; ++i) {
        if (PATH_FACE_IMAGES[i]) {
            gui->face_pixbufs[i] =
                gdk_pixbuf_new_from_file(PATH_FACE_IMAGES[i], NULL);
        } else {
            gui->face_pixbufs[i] = NULL;
        }
    }
    for (i = 0; i < 2; ++i) {
        gui->mine_pixbufs[i] =
            gdk_pixbuf_new_from_file(PATH_MINE_IMAGES[i], NULL);
    }
    for (i = NOT_STARTED; i < NUM_FLAGS_STATES; ++i) {
        gui->widget_state_images[i] =
            gtk_image_new_from_file(PATH_STATE_IMAGES[i]);
        g_object_ref(gui->widget_state_images[i]);
    }
}

/*
 * Destroy images and pixbufs.
 */
static void _um_gui_flags_destroy_resources(GUIFlags *gui) {
    int i;
    for (i = BLANK; i < PLAYER_FLAG + _UM_GUI_FLAGS_MAX_PLAYERS; ++i) {
        if (gui->face_pixbufs[i]) {
            g_object_unref(gui->face_pixbufs[i]);
        }
    }
    for (i = 0; i < 2; ++i) {
        g_object_unref(gui->mine_pixbufs[i]);
    }
    for (i = NOT_STARTED; i < NUM_FLAGS_STATES; ++i) {
        gtk_widget_destroy(gui->widget_state_images[i]);
    }
}

/*
 * Build the GUI (creates the top-level container and calls the other build
 * functions).
 */
static void _um_gui_flags_build(GUIFlags *gui, GtkWidget *window) {
    GtkWidget *right = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);
    gui->widget_container = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 0);
    _um_gui_flags_build_menu(gui);
    _um_gui_flags_build_players(gui);
    _um_gui_flags_build_controls(gui, right);
    _um_gui_flags_build_gboard(gui, right);
    gtk_box_pack_start(GTK_BOX(gui->widget_container), right, 1, 0, 0);
    _um_gui_flags_build_bomb_opts(gui, window);
    _um_gui_flags_build_player_opts(gui, window);
    gui->gparams = um_gui_params_create(
        window,
        UM_FLAGS_MIN.width, UM_FLAGS_MIN.height, UM_FLAGS_MIN.num_mines,
        MAX_PARAMETERS.width, MAX_PARAMETERS.height, MAX_PARAMETERS.num_mines
    );
}

/*
 * Build the bomb options dialog.
 */
static void _um_gui_flags_build_bomb_opts(GUIFlags *gui, GtkWidget *window) {
    GtkWidget *grid,
              *count_label,
              *range_label,
              *vbox;
    count_label = gtk_label_new_with_mnemonic(DIALOG_BOMB_COUNT);
    gui->widget_bomb_count = gtk_spin_button_new_with_range(0,
                            MAX_PARAMETERS.width * MAX_PARAMETERS.height, 1);
    range_label = gtk_label_new_with_mnemonic(DIALOG_BOMB_RANGE);
    gui->widget_bomb_range = gtk_spin_button_new_with_range(1,
                                                    MAX_PARAMETERS.width, 1);
    grid = gtk_grid_new();
    gtk_grid_attach(GTK_GRID(grid), count_label, 0, 0, 1, 1);
    gtk_grid_attach(GTK_GRID(grid), gui->widget_bomb_count, 1, 0, 1, 1);
    gtk_grid_attach(GTK_GRID(grid), range_label, 0, 1, 1, 1);
    gtk_grid_attach(GTK_GRID(grid), gui->widget_bomb_range, 1, 1, 1, 1);
    gtk_widget_show_all(grid);
    gui->widget_bomb_dialog = gtk_dialog_new_with_buttons(
        DIALOG_BOMB_TITLE,
        GTK_WINDOW(window),
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
        DIALOG_CANCEL, GTK_RESPONSE_CANCEL,
        DIALOG_OK, GTK_RESPONSE_OK,
        NULL
    );
    gtk_window_set_resizable(GTK_WINDOW(gui->widget_bomb_dialog), 0);
    vbox = gtk_dialog_get_content_area(GTK_DIALOG(gui->widget_bomb_dialog));
    gtk_container_add(GTK_CONTAINER(vbox), grid);
}

/*
 * Build the controls and pack them into the given box.
 */
static void _um_gui_flags_build_controls(GUIFlags *gui, GtkWidget *container) {
    GtkWidget *controls = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 0);
    gui->widget_mine_count = gtk_label_new(NULL);
    gui->widget_state = gtk_button_new();
    gui->widget_bomb_select = gtk_toggle_button_new();
    gtk_container_add(GTK_CONTAINER(gui->widget_bomb_select),
                        gtk_image_new_from_file(PATH_BOMB_IMAGE));
    g_signal_connect_swapped(gui->widget_state, "clicked",
                            G_CALLBACK(_um_gui_flags_reset), gui);
    g_signal_connect_swapped(gui->widget_bomb_select, "toggled",
                            G_CALLBACK(_um_gui_flags_toggle_bomb_select), gui);
    gtk_box_pack_start(GTK_BOX(controls), gui->widget_mine_count, 0, 0, 8);
    gtk_box_set_center_widget(GTK_BOX(controls), gui->widget_state);
    gtk_box_pack_end(GTK_BOX(controls), gui->widget_bomb_select, 0, 0, 8);
    gtk_box_pack_start(GTK_BOX(container), controls, 0, 0, 8);
}

/*
 * Build the GUI Board and pack it into the given container.
 */
static void _um_gui_flags_build_gboard(GUIFlags *gui, GtkWidget *container) {
    GUIBoardCallbacks callbacks = GBOARD_CALLBACKS;
    callbacks.data = gui;
    gui->gboard = um_gui_board_create(&callbacks, gui->face_pixbufs,
                                        gui->mine_pixbufs);
    gtk_box_pack_start(GTK_BOX(container),
                        um_gui_board_get_widget(gui->gboard), 1, 0, 0);
    gui->repopulate_gboard = 1;
}

/*
 * Build the game menu.
 */
static void _um_gui_flags_build_menu(GUIFlags *gui) {
    GtkWidget *allow_ties,
              *bomb_count_range,
              *bomb_ends_turn,
              *cover_on_done,
              *custom,
              *f_opts,
              *f_opts_menu,
              *no_expand_zero_tiles,
              *params,
              *params_menu,
              *players,
              *sep_f_opts,
              *sep_params,
              *standard;

    /* Create menu items. */
    f_opts = gtk_menu_item_new_with_mnemonic(MENU);
    params = gtk_menu_item_new_with_mnemonic(MENU_PARAMS);
    standard =
        gtk_radio_menu_item_new_with_mnemonic(NULL, MENU_PARAMS_STANDARD);
    sep_params = gtk_separator_menu_item_new();
    custom = gtk_radio_menu_item_new_with_mnemonic_from_widget(
                GTK_RADIO_MENU_ITEM(standard), MENU_PARAMS_CUSTOM);
    players = gtk_menu_item_new_with_mnemonic(MENU_PLAYERS);
    sep_f_opts = gtk_separator_menu_item_new();
    allow_ties = gtk_check_menu_item_new_with_mnemonic(MENU_ALLOW_TIES);
    bomb_count_range = gtk_menu_item_new_with_mnemonic(MENU_BOMB_COUNT_RANGE);
    bomb_ends_turn =
        gtk_check_menu_item_new_with_mnemonic(MENU_BOMB_ENDS_TURN);
    cover_on_done = gtk_check_menu_item_new_with_mnemonic(MENU_COVER_ON_DONE);
    no_expand_zero_tiles =
        gtk_check_menu_item_new_with_mnemonic(MENU_NO_EXPAND_ZERO_TILES);

    /* Set current option values. */
    if (gui->params.width == UM_FLAGS_DEFAULT.width &&
        gui->params.height == UM_FLAGS_DEFAULT.height &&
        gui->params.num_mines == UM_FLAGS_DEFAULT.num_mines) {
        gui->widget_preset = standard;
    } else {
        gui->widget_preset = custom;
    }
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(gui->widget_preset),
                                    TRUE);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(allow_ties),
                                    gui->options.allow_ties);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(bomb_ends_turn),
                                    gui->options.bomb_ends_turn);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(cover_on_done),
                                    gui->options.cover_on_done);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(no_expand_zero_tiles),
                                    gui->options.no_expand_zero_tiles);

    /* Create menus and fill them with menu items. */
    params_menu = gtk_menu_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(params_menu), standard);
    gtk_menu_shell_append(GTK_MENU_SHELL(params_menu), sep_params);
    gtk_menu_shell_append(GTK_MENU_SHELL(params_menu), custom);
    f_opts_menu = gtk_menu_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(f_opts_menu), params);
    gtk_menu_shell_append(GTK_MENU_SHELL(f_opts_menu), players);
    gtk_menu_shell_append(GTK_MENU_SHELL(f_opts_menu), sep_f_opts);
    gtk_menu_shell_append(GTK_MENU_SHELL(f_opts_menu), allow_ties);
    gtk_menu_shell_append(GTK_MENU_SHELL(f_opts_menu), bomb_count_range);
    gtk_menu_shell_append(GTK_MENU_SHELL(f_opts_menu), bomb_ends_turn);
    gtk_menu_shell_append(GTK_MENU_SHELL(f_opts_menu), cover_on_done);
    gtk_menu_shell_append(GTK_MENU_SHELL(f_opts_menu), no_expand_zero_tiles);

    /* Link menu items to submenus. */
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(params), params_menu);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(f_opts), f_opts_menu);

    /* Connect signal handlers. */
    g_signal_connect(standard, "activate",
                        G_CALLBACK(_um_gui_flags_params_standard), gui);
    g_signal_connect(custom, "activate",
                        G_CALLBACK(_um_gui_flags_params_custom), gui);
    g_signal_connect_swapped(players, "activate",
                        G_CALLBACK(_um_gui_flags_players), gui);
    g_signal_connect(allow_ties, "toggled",
                        G_CALLBACK(_um_gui_flags_toggle_allow_ties), gui);
    g_signal_connect_swapped(bomb_count_range, "activate",
                        G_CALLBACK(_um_gui_flags_bomb_options), gui);
    g_signal_connect(bomb_ends_turn, "toggled",
                        G_CALLBACK(_um_gui_flags_toggle_bomb_ends_turn), gui);
    g_signal_connect(cover_on_done, "toggled",
                        G_CALLBACK(_um_gui_flags_toggle_cover_on_done), gui);
    g_signal_connect(no_expand_zero_tiles, "toggled",
                G_CALLBACK(_um_gui_flags_toggle_no_expand_zero_tiles), gui);

    gui->widget_menu = f_opts;
}

/*
 * Build the players dialog.
 */
static void _um_gui_flags_build_player_opts(GUIFlags *gui, GtkWidget *window) {
    GtkWidget *grid,
              *vbox;
    int i;
    g_intern_static_string(DIALOG_PLAYERS_NONE);
    g_intern_static_string(DIALOG_PLAYERS_HOTSEAT);
    for (i = 0; i < NUM_AI_LEVELS; ++i) {
        g_intern_static_string(UM_FLAGS_AI_LEVEL_NAMES[i]);
    }
    grid = gtk_grid_new();
    for (i = 0; i < _UM_GUI_FLAGS_MAX_PLAYERS; ++i) {
        GtkWidget *label;
        char buffer[] = "...";
        if (i < 999) {
            sprintf(buffer, "%3d", i + 1);
        }
        label = gtk_label_new_with_mnemonic(buffer);
        gui->widget_player_selects[i] = gtk_combo_box_text_new();
        if (i > 1) {
            gtk_combo_box_text_append(
                GTK_COMBO_BOX_TEXT(gui->widget_player_selects[i]),
                DIALOG_PLAYERS_NONE,
                DIALOG_PLAYERS_NONE
            );
        }
        gtk_combo_box_text_append(
            GTK_COMBO_BOX_TEXT(gui->widget_player_selects[i]),
            DIALOG_PLAYERS_HOTSEAT,
            DIALOG_PLAYERS_HOTSEAT
        );
        if (i > 0) {
            int j;
            for (j = NO_AI + 1; j < NUM_AI_LEVELS; ++j) {
                gtk_combo_box_text_append(
                    GTK_COMBO_BOX_TEXT(gui->widget_player_selects[i]),
                    UM_FLAGS_AI_LEVEL_NAMES[j],
                    UM_FLAGS_AI_LEVEL_NAMES[j]
                );
            }
        }
        gtk_grid_attach(GTK_GRID(grid), label,
                        0, i, 1, 1);
        gtk_grid_attach(GTK_GRID(grid), gui->widget_player_selects[i],
                        1, i, 1, 1);
    }
    gtk_widget_show_all(grid);
    gui->widget_players_dialog = gtk_dialog_new_with_buttons(
        DIALOG_PLAYERS_TITLE,
        GTK_WINDOW(window),
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
        DIALOG_CANCEL, GTK_RESPONSE_CANCEL,
        DIALOG_OK, GTK_RESPONSE_OK,
        NULL
    );
    gtk_window_set_resizable(GTK_WINDOW(gui->widget_players_dialog), 0);
    vbox = gtk_dialog_get_content_area(GTK_DIALOG(gui->widget_players_dialog));
    gtk_container_add(GTK_CONTAINER(vbox), grid);
}

/*
 * Build the players table and pack it into the top-level container.
 */
static void _um_gui_flags_build_players(GUIFlags *gui) {
    GtkWidget *grid = gtk_grid_new(),
              *label;
    int i;
    label = gtk_label_new(NULL);
    gtk_grid_attach(GTK_GRID(grid), label, 0, 0, 1, 1);
    label = gtk_label_new(NULL);
    gtk_grid_attach(GTK_GRID(grid), label, 1, 0, 1, 1);
    label = gtk_label_new(NULL);
    gtk_grid_attach(GTK_GRID(grid), label, 2, 0, 1, 1);
    label = gtk_label_new(NULL);
    gtk_grid_attach(GTK_GRID(grid), label, 3, 0, 1, 1);
    label = gtk_label_new(PLAYERS_BOMBS);
    gtk_grid_attach(GTK_GRID(grid), label, 4, 0, 1, 1);
    label = gtk_label_new(PLAYERS_SCORE);
    gtk_grid_attach(GTK_GRID(grid), label, 5, 0, 1, 1);
    for (i = 0; i < _UM_GUI_FLAGS_MAX_PLAYERS; ++i) {
        gui->widget_player_wins[i] =
                                gtk_image_new_from_file(PATH_WINNER_IMAGE);
        gui->widget_player_prompts[i] =
                                gtk_image_new_from_file(PATH_PROMPT_IMAGE);
        gui->widget_player_icons[i] =
                                gtk_image_new_from_file(PATH_PLAYER_IMAGES[i]);
        gui->widget_player_labels[i] = gtk_label_new(NULL);
        gui->widget_player_bombs[i] = gtk_label_new(NULL);
        gui->widget_player_scores[i] = gtk_label_new(NULL);
        gtk_grid_attach(GTK_GRID(grid), gui->widget_player_wins[i],
                        0, i + 1, 1, 1);
        gtk_grid_attach(GTK_GRID(grid), gui->widget_player_prompts[i],
                        1, i + 1, 1, 1);
        gtk_grid_attach(GTK_GRID(grid), gui->widget_player_icons[i],
                        2, i + 1, 1, 1);
        gtk_grid_attach(GTK_GRID(grid), gui->widget_player_labels[i],
                        3, i + 1, 1, 1);
        gtk_grid_attach(GTK_GRID(grid), gui->widget_player_bombs[i],
                        4, i + 1, 1, 1);
        gtk_grid_attach(GTK_GRID(grid), gui->widget_player_scores[i],
                        5, i + 1, 1, 1);
    }
    gtk_widget_set_margin_top(grid, 8);
    gtk_widget_set_margin_bottom(grid, 8);
    gtk_widget_set_margin_start(grid, 8);
    gtk_widget_set_margin_end(grid, 21);
    gtk_grid_set_column_spacing(GTK_GRID(grid), 8);
    gtk_grid_set_row_spacing(GTK_GRID(grid), 8);
    gtk_box_pack_start(GTK_BOX(gui->widget_container), grid, 1, 0, 0);
}

/*
 * Initialize the game.
 */
static int _um_gui_flags_init_game(GUIFlags *gui) {
    FlagsCallbacks callbacks = FLAGS_CALLBACKS;
    callbacks.data = gui;
    gui->players = NULL;
    gui->state = NOT_STARTED;
    gui->turn_player = 0;
    gui->game = um_flags_create(&callbacks, &gui->options, &gui->params);
    if (!gui->game) {
        return 1;
    }
    um_flags_update(gui->game);
    return 0;
}

/*
 * Update the GUI board. This is a GSourceFunc.
 */
static int  _um_gui_flags_do_board(void *update) {
    BoardUpdate *b_update = (BoardUpdate *) update;
    if (b_update->gui->repopulate_gboard) {
        b_update->gui->repopulate_gboard = 0;
        if (um_gui_board_populate(b_update->gui->gboard, b_update->tiles,
                                    b_update->gui->params.width,
                                    b_update->gui->params.height)) {
            free(b_update);
            gtk_main_quit();
            return G_SOURCE_REMOVE;
        }
        um_gui_shrink();
    } else {
        um_gui_board_update(b_update->gui->gboard);
    }
    free(b_update);
    return G_SOURCE_REMOVE;
}

/*
 * Update the mine count. This is a GSourceFunc.
 */
static int  _um_gui_flags_do_mine_count(void *update) {
    MineCountUpdate *m_update = (MineCountUpdate *) update;
    char buffer[] = "...";
    if (m_update->mine_count < 1000) {
        sprintf(buffer, "%3d", m_update->mine_count);
    }
    gtk_label_set_text(GTK_LABEL(m_update->widget_mine_count), buffer);
    free(m_update);
    return G_SOURCE_REMOVE;
}

/*
 * Update the players table. This is a GSourceFunc.
 */
static int  _um_gui_flags_do_players(void *update) {
    PlayersUpdate *p_update = (PlayersUpdate *) update;
    int num_ai = 0,
        num_hotseat = 0,
        i;
    for (i = 0; i < _UM_GUI_FLAGS_MAX_PLAYERS; ++i) {
        char buffer_bombs[] = ".....",
             buffer_score[] = ".....";
        const char *label;
        gtk_widget_hide(p_update->gui->widget_player_wins[i]);
        if (i >= p_update->count) {
            gtk_widget_hide(p_update->gui->widget_player_icons[i]);
            gtk_label_set_text(
                GTK_LABEL(p_update->gui->widget_player_labels[i]), "");
            gtk_label_set_text(
                GTK_LABEL(p_update->gui->widget_player_bombs[i]), "");
            gtk_label_set_text(
                GTK_LABEL(p_update->gui->widget_player_scores[i]), "");
            continue;
        }
        if (p_update->players[i].ai > NO_AI) {
            label = PLAYERS_LABELS_AI[num_ai];
            ++num_ai;
        } else {
            label = PLAYERS_LABELS_HOTSEAT[num_hotseat];
            ++num_hotseat;
        }
        if (p_update->players[i].num_bombs < 100000) {
            sprintf(buffer_bombs, "%5d", p_update->players[i].num_bombs);
        }
        if (p_update->players[i].score < 100000) {
            sprintf(buffer_score, "%5d", p_update->players[i].score);
        }
        gtk_widget_show(p_update->gui->widget_player_icons[i]);
        gtk_label_set_text(
            GTK_LABEL(p_update->gui->widget_player_labels[i]), label);
        gtk_label_set_text(
            GTK_LABEL(p_update->gui->widget_player_bombs[i]), buffer_bombs);
        gtk_label_set_text(
            GTK_LABEL(p_update->gui->widget_player_scores[i]), buffer_score);
    }
    p_update->gui->players = p_update->players;
    free(p_update);
    return G_SOURCE_REMOVE;
}

/*
 * Transition between states. This is a GSourceFunc.
 */
static int  _um_gui_flags_do_state(void *update) {
    StateUpdate *s_update = (StateUpdate *) update;
    if (s_update->state != s_update->gui->state) {
        s_update->gui->state = s_update->state;
        if (s_update->state == DONE) {
            int num_players = s_update->gui->params.num_hotseat +
                                s_update->gui->params.num_ai,
                max = 0,
                i;
            for (i = 0; i < num_players; ++i) {
                if (s_update->gui->players[i].score > max) {
                    max = s_update->gui->players[i].score;
                }
            }
            for (i = 0; i < num_players; ++i) {
                gtk_widget_hide(s_update->gui->widget_player_prompts[i]);
                if (s_update->gui->players[i].score == max) {
                    gtk_widget_show(s_update->gui->widget_player_wins[i]);
                }
            }
        }
    }
    gtk_button_set_image(GTK_BUTTON(s_update->gui->widget_state),
                        s_update->gui->widget_state_images[s_update->state]);
    free(s_update);
    return G_SOURCE_REMOVE;
}

/*
 * Update the given tile on the GUI Board. This is a GSourceFunc.
 */
static int  _um_gui_flags_do_tile(void *update) {
    TileUpdate *t_update = (TileUpdate *) update;
    um_gui_board_update_tile(t_update->gboard, &t_update->coords);
    free(t_update);
    return G_SOURCE_REMOVE;
}

/*
 * Update the turn indicator. This is a GSourceFunc.
 */
static int  _um_gui_flags_do_turn(void *update) {
    TurnUpdate *t_update = (TurnUpdate *) update;
    int i;
    for (i = 0; i < _UM_GUI_FLAGS_MAX_PLAYERS; ++i) {
        if (i == t_update->turn_player) {
            gtk_widget_show(t_update->gui->widget_player_prompts[i]);
        } else {
            gtk_widget_hide(t_update->gui->widget_player_prompts[i]);
        }
    }
    t_update->gui->turn_player = t_update->turn_player;
    if (t_update->turn_player < t_update->gui->params.num_hotseat &&
        !t_update->gui->players[t_update->turn_player].num_bombs) {
        gtk_toggle_button_set_active(
            GTK_TOGGLE_BUTTON(t_update->gui->widget_bomb_select), 0);
    }
    free(t_update);
    return G_SOURCE_REMOVE;
}

/*
 * Request a board update. This is a FlagsGame callback function.
 */
static void _um_gui_flags_update_board(Tile **tiles, void *data) {
    GUIFlags *gui = (GUIFlags *) data;
    BoardUpdate *update = malloc(sizeof(BoardUpdate));
    if (!update) {
        return;
    }
    update->gui = gui;
    update->tiles = tiles;
    gdk_threads_add_idle(_um_gui_flags_do_board, update);
}

/*
 * Request a mine count update. This is a FlagsGame callback function.
 */
static void _um_gui_flags_update_mine_count(int mine_count, void *data) {
    GUIFlags *gui = (GUIFlags *) data;
    MineCountUpdate *update = malloc(sizeof(MineCountUpdate));
    if (!update) {
        return;
    }
    update->widget_mine_count = gui->widget_mine_count;
    update->mine_count = mine_count;
    gdk_threads_add_idle(_um_gui_flags_do_mine_count, update);
}

/*
 * Request a players update. This is a FlagsGame callback function.
 */
static void _um_gui_flags_update_players(const FlagsPlayer *players, int count,
                                            void *data) {
    GUIFlags *gui = (GUIFlags *) data;
    PlayersUpdate *update = malloc(sizeof(PlayersUpdate));
    if (!update) {
        return;
    }
    update->players = players;
    update->gui = gui;
    update->count = count;
    gdk_threads_add_idle(_um_gui_flags_do_players, update);
}

/*
 * Request a state update. This is a FlagsGame callback function.
 */
static void _um_gui_flags_update_state(FlagsState state, void *data) {
    GUIFlags *gui = (GUIFlags *) data;
    StateUpdate *update = malloc(sizeof(StateUpdate));
    if (!update) {
        return;
    }
    update->gui = gui;
    update->state = state;
    gdk_threads_add_idle(_um_gui_flags_do_state, update);
}

/*
 * Request a tile update. This is a FlagsGame callback function.
 */
static void _um_gui_flags_update_tile(const Tile *tile, void *data) {
    GUIFlags *gui = (GUIFlags *) data;
    TileUpdate *update = malloc(sizeof(TileUpdate));
    if (!update) {
        return;
    }
    update->gboard = gui->gboard;
    update->coords.x = tile->x;
    update->coords.y = tile->y;
    gdk_threads_add_idle(_um_gui_flags_do_tile, update);
}

/*
 * Request a turn update. This is a FlagsGame callback function.
 */
static void _um_gui_flags_update_turn(int player, void *data) {
    GUIFlags *gui = (GUIFlags *) data;
    TurnUpdate *update = malloc(sizeof(TurnUpdate));
    if (!update) {
        return;
    }
    update->gui = gui;
    update->turn_player = player;
    gdk_threads_add_idle(_um_gui_flags_do_turn, update);
}

/*
 * Set the game parameters. This can fail, in which case the application will
 * be terminated.
 */
static void _um_gui_flags_params(GUIFlags *gui) {
    gui->repopulate_gboard = 1;
    if (um_flags_parameters(gui->game, 0, &gui->params) > 0) {
        gtk_main_quit();
    }
}

/*
 * Depress the surrounding tiles if the bomb is selected and the game isn't
 * finished. This is a GUIBoard callback function.
 */
static void _um_gui_flags_mouse_enter(const Coords *coords, void *data) {
    GUIFlags *gui = (GUIFlags *) data;
    if (gui->state == DONE ||
        !gtk_toggle_button_get_active(
            GTK_TOGGLE_BUTTON(gui->widget_bomb_select))) {
        return;
    }
    um_gui_board_toggle_tiles(gui->gboard, coords, gui->options.bomb_range, 1,
                                BLANK);
}

/*
 * Return surrounding tiles to their normal state if the bomb is selected.
 * This is a GUIBoard callback function.
 */
static void _um_gui_flags_mouse_leave(const Coords *coords, void *data) {
    GUIFlags *gui = (GUIFlags *) data;
    if (!gtk_toggle_button_get_active(
            GTK_TOGGLE_BUTTON(gui->widget_bomb_select))) {
        return;
    }
    um_gui_board_toggle_tiles(gui->gboard, coords, gui->options.bomb_range, 0,
                                -1);
}

/*
 * Bomb or select the active tile. This is a GUIBoard callback function.
 */
static void _um_gui_flags_mouse_release(const Coords *coords, GdkEvent *event,
                                        void *data) {
    GUIFlags *gui = (GUIFlags *) data;
    if (coords->x == -1 || coords->y == -1 ||
        gui->state == DONE ||
        gui->turn_player >= gui->params.num_hotseat) {
        return;
    }
    if (gtk_toggle_button_get_active(
            GTK_TOGGLE_BUTTON(gui->widget_bomb_select))) {
        int b = gui->players[gui->turn_player].num_bombs;
        um_gui_board_toggle_tiles(gui->gboard, coords, gui->options.bomb_range,
                                    0, -1);
        if (!um_flags_bomb(gui->game, gui->turn_player, coords) && b == 1) {
            gtk_toggle_button_set_active(
                GTK_TOGGLE_BUTTON(gui->widget_bomb_select), 0);
        }
    } else {
        um_flags_select(gui->game, gui->turn_player, coords);
    }
}

/*
 * Run the bomb options dialog. If canceled, reset the form; if affirmed, set
 * the new bomb options. Signal handler for the 'Bomb Count / Range' menu item.
 */
static void _um_gui_flags_bomb_options(GUIFlags *gui) {
    int response;
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui->widget_bomb_count),
                                (double) gui->options.num_bombs);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui->widget_bomb_range),
                                (double) gui->options.bomb_range);
    response = gtk_dialog_run(GTK_DIALOG(gui->widget_bomb_dialog));
    gtk_widget_hide(gui->widget_bomb_dialog);
    if (response != GTK_RESPONSE_OK) {
        return;
    }
    gui->options.num_bombs = gtk_spin_button_get_value_as_int(
                                GTK_SPIN_BUTTON(gui->widget_bomb_count));
    gui->options.bomb_range = gtk_spin_button_get_value_as_int(
                                GTK_SPIN_BUTTON(gui->widget_bomb_range));
    um_flags_options(gui->game, 0, &gui->options);
}

/*
 * Run the game parameters dialog. If canceled, reset the form; if affirmed,
 * set the new game parameters. Signal handler for the 'Custom' menu item.
 */
static void _um_gui_flags_params_custom(GtkWidget *menu_item, GUIFlags *gui) {
    if (!gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menu_item))) {
        return;
    }
    if (um_gui_params_prompt(gui->gparams, &gui->params.width,
                                &gui->params.height, &gui->params.num_mines)) {
        _um_gui_flags_params(gui);
        gui->widget_preset = menu_item;
    } else {
        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(gui->widget_preset),
                                        TRUE);
    }
}

/*
 * Set the game parameters to the standard size. Signal handler for the
 * standard menu item.
 */
static void _um_gui_flags_params_standard(GtkWidget *menu_item,
                                            GUIFlags *gui) {
    if (gui->widget_preset != menu_item &&
        gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menu_item))) {
        gui->params.width = UM_FLAGS_DEFAULT.width;
        gui->params.height = UM_FLAGS_DEFAULT.height;
        gui->params.num_mines = UM_FLAGS_DEFAULT.num_mines;
        _um_gui_flags_params(gui);
        gui->widget_preset = menu_item;
    }
}

/*
 * Run the players dialog. If canceled, reset the form; if affirmed, set the
 * new players. Signal handler for the 'Players' menu item.
 */
static void _um_gui_flags_players(GUIFlags *gui) {
    int num_hotseat = gui->params.num_hotseat,
        response,
        i;
    for (i = 0; i < _UM_GUI_FLAGS_MAX_PLAYERS; ++i) {
        const char *id = DIALOG_PLAYERS_NONE;
        if (i < num_hotseat) {
            id = DIALOG_PLAYERS_HOTSEAT;
        } else if (i < num_hotseat + gui->params.num_ai) {
            id = UM_FLAGS_AI_LEVEL_NAMES[
                gui->params.ai_levels[i - num_hotseat]
            ];
        }
        gtk_combo_box_set_active_id(
                            GTK_COMBO_BOX(gui->widget_player_selects[i]), id);
    }
    response = gtk_dialog_run(GTK_DIALOG(gui->widget_players_dialog));
    gtk_widget_hide(gui->widget_players_dialog);
    if (response != GTK_RESPONSE_OK) {
        return;
    }
    gui->params.num_hotseat = 0;
    gui->params.num_ai = 0;
    for (i = 0; i < _UM_GUI_FLAGS_MAX_PLAYERS; ++i) {
        int j;
        const char *id = gtk_combo_box_get_active_id(
                            GTK_COMBO_BOX(gui->widget_player_selects[i]));
        if (id == DIALOG_PLAYERS_NONE) {
            continue;
        }
        if (id == DIALOG_PLAYERS_HOTSEAT) {
            ++gui->params.num_hotseat;
            continue;
        }
        for (j = NO_AI + 1; j < NUM_AI_LEVELS; ++j) {
            if (id == UM_FLAGS_AI_LEVEL_NAMES[j]) {
                gui->params.ai_levels[gui->params.num_ai] = j;
                ++gui->params.num_ai;
                break;
            }
        }
    }
    _um_gui_flags_params(gui);
}

/*
 * Reset the game. Signal handler for the reset button.
 */
static void _um_gui_flags_reset(GUIFlags *gui) {
    if (gui->state != NOT_STARTED) {
        um_flags_reset(gui->game, 0);
    }
}

/*
 * Set the allow ties option. Signal handler for the 'Allow Ties' menu item.
 */
static void _um_gui_flags_toggle_allow_ties(GtkWidget *menu_item,
                                            GUIFlags *gui) {
    gui->options.allow_ties = gtk_check_menu_item_get_active(
        GTK_CHECK_MENU_ITEM(menu_item)
    );
    um_flags_options(gui->game, 0, &gui->options);
}

/*
 * Set the bomb ends turn option. Signal handler for the 'Bomb Ends Turn' menu
 * item.
 */
static void _um_gui_flags_toggle_bomb_ends_turn(GtkWidget *menu_item,
                                                GUIFlags *gui) {
    gui->options.bomb_ends_turn = gtk_check_menu_item_get_active(
        GTK_CHECK_MENU_ITEM(menu_item)
    );
    um_flags_options(gui->game, 0, &gui->options);
}

/*
 * Raise tiles when the bomb is deselected. Signal handler for the bomb toggle
 * button.
 */
static void _um_gui_flags_toggle_bomb_select(GUIFlags *gui) {
    if (gtk_toggle_button_get_active(
            GTK_TOGGLE_BUTTON(gui->widget_bomb_select))) {
        return;
    }
    um_gui_board_toggle_all(gui->gboard, 0, -1);
}

/*
 * Set the cover on done option. Signal handler for the 'Cover on Done' menu
 * item.
 */
static void _um_gui_flags_toggle_cover_on_done(GtkWidget *menu_item,
                                                GUIFlags *gui) {
    gui->options.cover_on_done = gtk_check_menu_item_get_active(
        GTK_CHECK_MENU_ITEM(menu_item)
    );
    um_flags_options(gui->game, 0, &gui->options);
}

/*
 * Set the no expand zero tiles option. Signal handler for the 'No Expand Zero
 * Tiles' menu item.
 */
static void _um_gui_flags_toggle_no_expand_zero_tiles(GtkWidget *menu_item,
                                                        GUIFlags *gui) {
    gui->options.no_expand_zero_tiles = gtk_check_menu_item_get_active(
        GTK_CHECK_MENU_ITEM(menu_item)
    );
    um_flags_options(gui->game, 0, &gui->options);
}

/*
 * Public functions.
 */

/*
 * Initialize the interface.
 */
GUIFlags * um_gui_flags_init(GtkWidget *window, config_setting_t *config) {
    GUIFlags *gui;
    gui = malloc(sizeof(GUIFlags));
    if (!gui) {
        return NULL;
    }
    gui->config = config;
    _um_gui_flags_config_load(gui);
    _um_gui_flags_load_resources(gui);
    _um_gui_flags_build(gui, window);
    if (_um_gui_flags_init_game(gui)) {
        _um_gui_flags_destroy_resources(gui);
        free(gui);
        return NULL;
    }
    return gui;
}

/*
 * Destroy the interface.
 */
void um_gui_flags_destroy(GUIFlags *gui) {
    um_flags_config_save(gui->config, &gui->options, &gui->params,
                            _UM_GUI_FLAGS_MAX_PLAYERS);
    um_flags_destroy(gui->game);
    um_gui_params_destroy(gui->gparams);
    um_gui_board_destroy(gui->gboard);
    _um_gui_flags_destroy_resources(gui);
    free(gui);
}

/*
 * Get the 'Flags Options' menu widget (GtkMenuItem).
 */
GtkWidget * um_gui_flags_get_menu(GUIFlags *gui) {
    return gui->widget_menu;
}

/*
 * Get the top-level container widget (GtkBox).
 */
GtkWidget * um_gui_flags_get_ui(GUIFlags *gui) {
    return gui->widget_container;
}
