/*
 * unmine/gui_params.c
 *
 * Copyright 2017 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "gui_params.h"

#include "stdlib.h"

/*
 * Strings.
 */
static const char *PARAMS_TITLE = "Set Mines / Size";
static const char *PARAMS_WIDTH = "_Width";
static const char *PARAMS_HEIGHT = "_Height";
static const char *PARAMS_MINES = "# _Mines";
static const char *PARAMS_CANCEL = "Cancel";
static const char *PARAMS_OK = "OK";

/*
 * Types.
 */

struct GUIParams {
    GtkWidget *dialog,
              *width,
              *height,
              *num_mines;
};

/*
 * Private functions.
 */

/*
 * Build the parameters dialog.
 */
static void _um_gui_params_build(GUIParams *params, GtkWidget *window,
                                    int width_min, int height_min,
                                    int num_mines_min, int width_max,
                                    int height_max, int num_mines_max) {
    GtkWidget *grid,
              *height_label,
              *num_mines_label,
              *vbox,
              *width_label;
    width_label = gtk_label_new_with_mnemonic(PARAMS_WIDTH);
    params->width = gtk_spin_button_new_with_range(width_min, width_max, 1);
    height_label = gtk_label_new_with_mnemonic(PARAMS_HEIGHT);
    params->height = gtk_spin_button_new_with_range(height_min, height_max, 1);
    num_mines_label = gtk_label_new_with_mnemonic(PARAMS_MINES);
    params->num_mines = gtk_spin_button_new_with_range(num_mines_min,
                                                        num_mines_max, 1);
    grid = gtk_grid_new();
    gtk_grid_attach(GTK_GRID(grid), width_label, 0, 0, 1, 1);
    gtk_grid_attach(GTK_GRID(grid), params->width, 1, 0, 1, 1);
    gtk_grid_attach(GTK_GRID(grid), height_label, 0, 1, 1, 1);
    gtk_grid_attach(GTK_GRID(grid), params->height, 1, 1, 1, 1);
    gtk_grid_attach(GTK_GRID(grid), num_mines_label, 0, 2, 1, 1);
    gtk_grid_attach(GTK_GRID(grid), params->num_mines, 1, 2, 1, 1);
    gtk_widget_show_all(grid);
    params->dialog = gtk_dialog_new_with_buttons(
        PARAMS_TITLE,
        GTK_WINDOW(window),
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
        PARAMS_CANCEL, GTK_RESPONSE_CANCEL,
        PARAMS_OK, GTK_RESPONSE_OK,
        NULL
    );
    gtk_window_set_resizable(GTK_WINDOW(params->dialog), 0);
    vbox = gtk_dialog_get_content_area(GTK_DIALOG(params->dialog));
    gtk_container_add(GTK_CONTAINER(vbox), grid);
}

/*
 * Public functions.
 */

/*
 * Create a new GUIParams dialog with the given minimum and maximum values.
 * Returns a pointer to the new dialog, or NULL.
 */
GUIParams * um_gui_params_create(GtkWidget *window, int width_min,
                                    int height_min, int num_mines_min,
                                    int width_max, int height_max,
                                    int num_mines_max) {
    GUIParams *params = malloc(sizeof(GUIParams));
    if (!params) {
        return NULL;
    }
    _um_gui_params_build(params, window, width_min, height_min, num_mines_min,
                            width_max, height_max, num_mines_max);
    return params;
}

/*
 * Destroy the GUIParams dialog.
 */
void um_gui_params_destroy(GUIParams *params) {
    free(params);
}

/*
 * Prompt the user for parameters. Returns 1 if parameters change, otherwise 0.
 */
int um_gui_params_prompt(GUIParams *params, int *width, int *height,
                            int *num_mines) {
    int response;
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(params->width), (double) *width);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(params->height),
                                (double) *height);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(params->num_mines),
                                (double) *num_mines);
    response = gtk_dialog_run(GTK_DIALOG(params->dialog));
    gtk_widget_hide(params->dialog);
    if (response == GTK_RESPONSE_OK) {
        *width = gtk_spin_button_get_value_as_int(
                                        GTK_SPIN_BUTTON(params->width));
        *height = gtk_spin_button_get_value_as_int(
                                        GTK_SPIN_BUTTON(params->height));
        *num_mines = gtk_spin_button_get_value_as_int(
                                        GTK_SPIN_BUTTON(params->num_mines));
        return 1;
    }
    return 0;
}
