/*
 * unmine/tui.c
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#define _POSIX_C_SOURCE 199506L
#define _XOPEN_SOURCE 1
#define _XOPEN_SOURCE_EXTENDED 1

#include <locale.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <libconfig.h>
#include <ncursesw/curses.h>

#include "board.h"
#include "coords.h"
#include "tile.h"
#include "tui_classic.h"
#include "tui_common.h"
#include "tui_flags.h"
#include "unmine.h"

/*
 * Menu item counts.
 */
static const int NUM_SELECTIONS_MAIN = 2;

/*
 * Splash screen parameters.
 */
static const int SPLASH_BOARD_WIDTH = 57;
static const int SPLASH_BOARD_HEIGHT = 7;
static const Coords SPLASH_BOARD_MINES_COORDS[] = {
    {5, 1}, {6, 1}, {9, 1}, {10, 1}, {14, 1}, {15, 1}, {16, 1}, {17, 1},
    {18, 1}, {19, 1}, {23, 1}, {24, 1}, {25, 1}, {26, 1}, {27, 1}, {28, 1},
    {29, 1}, {30, 1}, {31, 1}, {32, 1}, {36, 1}, {37, 1}, {41, 1}, {42, 1},
    {43, 1}, {44, 1}, {45, 1}, {46, 1}, {50, 1}, {51, 1}, {52, 1}, {53, 1},
    {54, 1}, {55, 1},

    {4, 2}, {5, 2}, {8, 2}, {9, 2}, {13, 2}, {14, 2}, {17, 2}, {18, 2},
    {22, 2}, {23, 2}, {26, 2}, {27, 2}, {30, 2}, {31, 2}, {35, 2}, {36, 2},
    {40, 2}, {41, 2}, {44, 2}, {45, 2}, {49, 2}, {50, 2},

    {3, 3}, {4, 3}, {7, 3}, {8, 3}, {12, 3}, {13, 3}, {16, 3}, {17, 3},
    {21, 3}, {22, 3}, {25, 3}, {26, 3}, {29, 3}, {30, 3}, {34, 3}, {35, 3},
    {39, 3}, {40, 3}, {43, 3}, {44, 3}, {48, 3}, {49, 3}, {50, 3}, {51, 3},
    {52, 3}, {53, 3},

    {2, 4}, {3, 4}, {6, 4}, {7, 4}, {11, 4}, {12, 4}, {15, 4}, {16, 4},
    {20, 4}, {21, 4}, {24, 4}, {25, 4}, {28, 4}, {29, 4}, {33, 4}, {34, 4},
    {38, 4}, {39, 4}, {42, 4}, {43, 4}, {47, 4}, {48, 4},

    {1, 5}, {2, 5}, {3, 5}, {4, 5}, {5, 5}, {6, 5}, {10, 5}, {11, 5}, {14, 5},
    {15, 5}, {19, 5}, {20, 5}, {23, 5}, {24, 5}, {27, 5}, {28, 5}, {32, 5},
    {33, 5}, {37, 5}, {38, 5}, {41, 5}, {42, 5}, {46, 5}, {47, 5}, {48, 5},
    {49, 5}, {50, 5}, {51, 5}
};
static const int SPLASH_BOARD_MINES_COORDS_LENGTH = 132;

/*
 * Strings.
 */
static const char *CONFIG_FILE = ".unmine-tui";
static const char *CONFIG_NAME_CLASSIC = "classic";
static const char *CONFIG_NAME_FLAGS = "flags";
static const char *CONFIG_NAME_GAME = "game";
static const char *TEXT_MAIN_LICENSE = "free software - redistributable under the GNU GPLv2+";
static const char * const TEXT_MAIN_DIRECTIONS[] = {
    "[up/down] move selection",
    "[enter]   select",
    "[q/esc]   quit"
};
static const int TEXT_MAIN_DIRECTIONS_LENGTH = 3;
static const char *TEXT_MAIN_GAME_CLASSIC = "Classic";
static const char *TEXT_MAIN_GAME_FLAGS = "Flags";
static const char *TEXT_MAIN_PROMPT = "Select a game:";

/*
 * Private functions.
 */

/*
 * Initialize config object, read the config file, and add missing options.
 */
static void _um_tui_config_init(config_t *config) {
    config_setting_t *root;
    config_init(config);
    config_read_file(config, CONFIG_FILE);
    root = config_root_setting(config);
    if (!config_setting_get_member(root, CONFIG_NAME_GAME)) {
        config_setting_add(root, CONFIG_NAME_GAME, CONFIG_TYPE_INT);
    }
    if (!config_setting_get_member(root, CONFIG_NAME_CLASSIC)) {
        config_setting_add(root, CONFIG_NAME_CLASSIC, CONFIG_TYPE_GROUP);
    }
    if (!config_setting_get_member(root, CONFIG_NAME_FLAGS)) {
        config_setting_add(root, CONFIG_NAME_FLAGS, CONFIG_TYPE_GROUP);
    }
}

/*
 * Save the config file and destroy the config object.
 */
static void _um_tui_config_destroy(config_t *config, int game) {
    config_setting_t *root = config_root_setting(config);
    config_setting_t *setting_game = config_setting_get_member(root,
                                                            CONFIG_NAME_GAME);
    if (setting_game) {
        config_setting_set_int(setting_game, game);
    }
    config_write_file(config, CONFIG_FILE);
    config_destroy(config);
}

/*
 * Initialize ncurses colour pairs.
 */
static void _um_tui_init_colour_pairs(void) {
    if (has_colors()) {
        /* colours for revealed tiles without mines */
        init_pair(1,  COLOR_BLACK,   COLOR_BLACK);
        init_pair(2,  COLOR_BLUE,    COLOR_BLACK);
        init_pair(3,  COLOR_GREEN,   COLOR_BLACK);
        init_pair(4,  COLOR_RED,     COLOR_BLACK);
        init_pair(5,  COLOR_MAGENTA, COLOR_BLACK);
        init_pair(6,  COLOR_YELLOW,  COLOR_BLACK);
        init_pair(7,  COLOR_CYAN,    COLOR_BLACK);
        init_pair(8,  COLOR_WHITE,   COLOR_BLACK);
        init_pair(9,  COLOR_RED,     COLOR_BLACK);
        /* colours for selected revealed tiles without mines */
        init_pair(10, COLOR_YELLOW,  COLOR_YELLOW);
        init_pair(11, COLOR_BLUE,    COLOR_YELLOW);
        init_pair(12, COLOR_GREEN,   COLOR_YELLOW);
        init_pair(13, COLOR_RED,     COLOR_YELLOW);
        init_pair(14, COLOR_MAGENTA, COLOR_YELLOW);
        init_pair(15, COLOR_BLACK,   COLOR_YELLOW);
        init_pair(16, COLOR_CYAN,    COLOR_YELLOW);
        init_pair(17, COLOR_WHITE,   COLOR_YELLOW);
        init_pair(18, COLOR_RED,     COLOR_YELLOW);

        /* colour for classic: tile with revealed mine */
        init_pair(19, COLOR_RED,     COLOR_BLACK);
        /* colour for classic: selected tile with revealed mine */
        init_pair(20, COLOR_RED,     COLOR_YELLOW);

        /* colour for classic: tile with exploded mine */
        init_pair(21, COLOR_BLACK,   COLOR_RED);
        /* colour for classic: selected tile with exploded mine */
        init_pair(22, COLOR_BLACK,   COLOR_YELLOW);

        /* colour for classic: tile with mine flag */
        init_pair(23, COLOR_RED,     COLOR_WHITE);
        /* colour for classic: selected tile with mine flag */
        init_pair(24, COLOR_RED,     COLOR_YELLOW);

        /* colour for classic: tile with question mark */
        init_pair(25, COLOR_BLACK,   COLOR_WHITE);
        /* colour for classic: selected tile with question mark */
        init_pair(26, COLOR_BLACK,   COLOR_YELLOW);

        /* colour for classic: tile with incorrect mine flag */
        init_pair(27, COLOR_BLACK,   COLOR_RED);
        /* colour for classic: selected tile with incorrect mine flag */
        init_pair(28, COLOR_BLACK,   COLOR_YELLOW);

        /* colour for unrevealed tile */
        init_pair(29, COLOR_WHITE,   COLOR_WHITE);
        /* colour for selected unrevealed tile */
        init_pair(30, COLOR_WHITE,   COLOR_YELLOW);

        /* colours for flags: flagged mines */
        init_pair(31, COLOR_BLUE,    COLOR_BLACK);
        init_pair(32, COLOR_RED,     COLOR_BLACK);
        init_pair(33, COLOR_GREEN,   COLOR_BLACK);
        init_pair(34, COLOR_YELLOW,  COLOR_BLACK);
        init_pair(35, COLOR_MAGENTA, COLOR_BLACK);
        init_pair(36, COLOR_CYAN,    COLOR_BLACK);
        init_pair(37, COLOR_WHITE,   COLOR_BLACK);
        /* colours for flags: selected flagged mines */
        init_pair(38, COLOR_BLUE,    COLOR_YELLOW);
        init_pair(39, COLOR_RED,     COLOR_YELLOW);
        init_pair(40, COLOR_GREEN,   COLOR_YELLOW);
        init_pair(41, COLOR_BLACK,   COLOR_YELLOW);
        init_pair(42, COLOR_MAGENTA, COLOR_YELLOW);
        init_pair(43, COLOR_CYAN,    COLOR_YELLOW);
        init_pair(44, COLOR_WHITE,   COLOR_YELLOW);
    }
}

/*
 * Draw the main menu.
 */
static void _um_tui_main_draw(Tile **tiles) {
    int i, j,
        start_x = UM_TUI_CENTER(COLS, SPLASH_BOARD_WIDTH),
        start_y = (LINES - SPLASH_BOARD_HEIGHT) / 4;
    attr_off(A_BOLD, NULL);
    for (i = 0; i < SPLASH_BOARD_WIDTH; ++i) {
        for (j = 0; j < SPLASH_BOARD_HEIGHT; ++j) {
            char face;
            int pair_index;
            if (tiles[i][j].mined) {
                face = '>';
                pair_index = 23;
            } else {
                face = ' ';
                if (tiles[i][j].count) {
                    if (rand() < RAND_MAX / 6.0) {
                        face = tiles[i][j].count + '0';
                    }
                }
                pair_index = tiles[i][j].count + 1;
            }
            attr_on(COLOR_PAIR(pair_index), NULL);
            mvaddch(start_y + tiles[i][j].y, start_x + tiles[i][j].x, face);
            attr_off(COLOR_PAIR(pair_index), NULL);
        }
    }
    mvaddstr(
        LINES / 2 + 2,
        UM_TUI_CENTER(COLS, strlen(TEXT_MAIN_PROMPT)),
        TEXT_MAIN_PROMPT
    );
    attr_on(A_BOLD, NULL);
    mvaddstr(
        LINES / 2 + 4,
        UM_TUI_CENTER(COLS, strlen(TEXT_MAIN_GAME_CLASSIC)),
        TEXT_MAIN_GAME_CLASSIC
    );
    mvaddstr(
        LINES / 2 + 6,
        UM_TUI_CENTER(COLS, strlen(TEXT_MAIN_GAME_FLAGS)),
        TEXT_MAIN_GAME_FLAGS
    );
}

/*
 * Draw Copyright and license information in the top right corner.
 */
static void _um_tui_main_draw_copyright(void) {
    attr_off(A_BOLD, NULL);
    mvaddstr(0, COLS - strlen(UM_COPYRIGHT),      UM_COPYRIGHT);
    mvaddstr(1, COLS - strlen(TEXT_MAIN_LICENSE), TEXT_MAIN_LICENSE);
    attr_on(A_BOLD, NULL);
}

/*
 * Draw the cursor on the main menu.
 */
static void _um_tui_main_draw_selection(int selection, int clear_old) {
    int x = UM_TUI_CENTER(COLS, strlen(TEXT_MAIN_GAME_CLASSIC)) - 2,
        y_start = LINES / 2 + 4;
    if (clear_old) {
        mvaddch(y_start,     x, ' ');
        mvaddch(y_start + 2, x, ' ');
    }
    mvaddch(y_start + selection * 2, x, '>');
}

/*
 * Main menu.
 */
static void _um_tui_main(void) {
    int input = 0,
        selection = 0;
    Board *board = um_board_create(SPLASH_BOARD_WIDTH, SPLASH_BOARD_HEIGHT);
    Tile **tiles = um_board_tiles(board);
    config_t config;
    config_setting_t *root;
    _um_tui_config_init(&config);
    root = config_root_setting(&config);
    config_setting_lookup_int(root, CONFIG_NAME_GAME, &selection);
    um_board_mine(board, SPLASH_BOARD_MINES_COORDS,
                    SPLASH_BOARD_MINES_COORDS_LENGTH);
    do {
        int new_selection = 0,
            redraw = um_tui_is_size_changed();
        switch(input) {
            case KEY_UP:
                --selection;
                new_selection = 1;
                break;
            case KEY_DOWN:
                ++selection;
                new_selection = 1;
                break;
            case 10:
            case 13:
            case KEY_B2:
                if (!selection) {
                    um_tui_classic(
                        config_setting_get_member(root, CONFIG_NAME_CLASSIC));
                } else {
                    um_tui_flags(
                        config_setting_get_member(root, CONFIG_NAME_FLAGS));
                }
                redraw = 1;
                break;
        }
        if (new_selection) {
            selection = um_tui_positive_modulo(selection, NUM_SELECTIONS_MAIN);
        }
        if (redraw) {
            um_tui_draw_meta(NULL, NULL, TEXT_MAIN_DIRECTIONS,
                                TEXT_MAIN_DIRECTIONS_LENGTH);
            _um_tui_main_draw(tiles);
            _um_tui_main_draw_copyright();
            _um_tui_main_draw_selection(selection, 0);
        } else if (new_selection) {
            _um_tui_main_draw_selection(selection, 1);
        }
        if (redraw || new_selection) {
            refresh();
        }
        input = getch();
    } while (!um_tui_is_quit_key(input));
    _um_tui_config_destroy(&config, selection);
    um_board_destroy(board);
}

/*
 * Public functions.
 */

/*
 * Start the interface.
 */
int main(void) {
    setlocale(LC_ALL, "");
    srand(time(NULL));
    initscr();
    start_color();
    _um_tui_init_colour_pairs();
    curs_set(0);
    noecho();
    cbreak();
    #ifdef NCURSES_VERSION
        set_escdelay(0);
    #endif /* NCURSES_VERSION */
    halfdelay(1);
    keypad(stdscr, TRUE);
    attr_on(A_BOLD, NULL);
    _um_tui_main();
    endwin();
    return 0;
}
