/*
 * unmine/tui_classic.c
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#define _POSIX_C_SOURCE 199506L
#define _XOPEN_SOURCE 1
#define _XOPEN_SOURCE_EXTENDED 1

#include "tui_classic.h"

#include <string.h>

#include <ncursesw/curses.h>

#include "classic.h"
#include "classic_config.h"
#include "coords.h"
#include "tile.h"
#include "tui_common.h"

/*
 * Menu item counts.
 */
static const int NUM_SELECTIONS_WINMODE = 4;

/*
 * Strings.
 */

static const char *TEXT_STATE_CLEAR = "         ";
static const char * const TEXT_STATES[] = {
    "READY", "GO!", "GAME OVER", "WINNER!"
};
static const char *TEXT_OPTIONS_TITLE = "OPTIONS";
static const char *TEXT_OPTIONS_WINMODE = "[w]in mode";
static const char * const TEXT_OPTIONS_WINMODE_VALUES[] = {
    "reveal", "flags ", "either", "both  "
};
static const char *TEXT_OPTIONS_COVERONLOSS = "[c]over on loss";
static const char *TEXT_OPTIONS_NOEXPANDZEROTILES = "no e[x]pand zero tiles";
static const char *TEXT_OPTIONS_ZOOM = "[z]oom";
static const char * const TEXT_OPTIONS_VALUES[] = {"off", "on "};
static const char *TEXT_TITLE = "Classic Game";
static const char * const TEXT_DIRECTIONS[] = {
    "[arrows] move selection",
    "[space]  reveal",
    "[f]      mine flag",
    "[e]      question mark",
    "[v]      chord",
    "[r]      reset",
    "[m/s]    change # mines / size",
    "[q/esc]  quit"
};
static const int TEXT_DIRECTIONS_LENGTH = 8;
static const char *TEXT_WINMODE_TITLE = "Win Mode";
static const char * const TEXT_WINMODE_DIRECTIONS[] = {
    "[up/down]    move selection",
    "[enter]      ok",
    "[q/esc]      cancel"
};
static const int TEXT_WINMODE_DIRECTIONS_LENGTH = 3;
static const char *TEXT_WINMODE_PROMPT = "Win the game by:";
static const char *TEXT_WINMODE_BOTH = "Both";
static const char *TEXT_WINMODE_EITHER = "Either";
static const char *TEXT_WINMODE_FLAGS = "Flagging all mines";
static const char *TEXT_WINMODE_REVEAL = "Revealing all tiles";


/*
 * Types.
 */

typedef struct {
    Coords selection;
    ClassicParameters params;
    int zoom;
} CallbackData;


/*
 * Private functions.
 */

/*
 * Load configuration.
 */
static void _um_tui_classic_config_load(config_setting_t *config,
                                        ClassicOptions *options,
                                        ClassicParameters *params, int *zoom) {
    um_classic_config_load(config, options, params);
    um_tui_config_load_zoom(config, zoom);
}

/*
 * Save configuration.
 */
static void _um_tui_classic_config_save(config_setting_t *config,
                                        ClassicOptions *options,
                                        ClassicParameters *params, int zoom) {
    um_classic_config_save(config, options, params);
    um_tui_config_save_zoom(config, zoom);
}

/*
 * Draw the given Tile.
 */
static void _um_tui_classic_draw_tile(Tile *tile, void *data) {
    CallbackData *cb_data = (CallbackData *) data;
    int pair_index = 1,
        selected = tile->x == cb_data->selection.x &&
                    tile->y == cb_data->selection.y,
        x = UM_TUI_BOARD_X(cb_data->params.width, cb_data->zoom) +
                (cb_data->zoom + 1) * tile->x,
        y = UM_TUI_BOARD_Y(cb_data->params.height, cb_data->zoom) +
                (cb_data->zoom + 1) * tile->y;
    chtype face = ' ';
    if (tile->revealed) {
        if (tile->mined) {
            /* revealed mined tile */
            face = '*';
            if (tile->last) {
                pair_index = 21 + selected;
            } else {
                pair_index = 19 + selected;
            }
        } else {
            /* revealed non-mined tile */
            if (tile->count) {
                face = tile->count + '0';
            }
            pair_index = tile->count + 1 + selected * 9;
        }
    } else if (tile->face == MINE_FLAG) {
        /* flagged tile */
        face = '>';
        pair_index = 23 + selected;
        attr_off(A_BOLD, NULL);
    } else if (tile->face == QUESTION_MARK) {
        /* question mark tile */
        face = '?';
        pair_index = 25 + selected;
    } else if (tile->face == INCORRECT_FLAG) {
        /* incorrect mine flag tile */
        face = '>';
        pair_index = 27 + selected;
        attr_off(A_BOLD, NULL);
    } else {
        /* blank unrevealed tile */
        pair_index = 29 + selected;
    }
    attr_on(COLOR_PAIR(pair_index), NULL);
    mvaddch(y, x, face);
    attr_off(COLOR_PAIR(pair_index), NULL);
    if (tile->face == MINE_FLAG || tile->face == INCORRECT_FLAG) {
        attr_on(A_BOLD, NULL);
    }
}

/*
 * Draw the whole board (wraps _um_tui_classic_draw_tile).
 */
static void _um_tui_classic_draw_board(Tile **tiles, void *data) {
    CallbackData *cb_data = (CallbackData *) data;
    int i, j;
    for (i = 0; i < cb_data->params.width; ++i) {
        for (j = 0; j < cb_data->params.height; ++j) {
            _um_tui_classic_draw_tile(&tiles[i][j], data);
        }
    }
}

/*
 * Draw the flag count.
 */
static void _um_tui_classic_draw_flag_count(int flag_count, void *data) {
    CallbackData *cb_data = (CallbackData *) data;
    char buffer[] = "...";
    if (flag_count > -100 && flag_count < 1000) {
        sprintf(buffer, "%3d", flag_count);
    }
    mvaddstr(
        UM_TUI_BOARD_Y(cb_data->params.height, cb_data->zoom) - 2,
        UM_TUI_BOARD_X(cb_data->params.width, cb_data->zoom),
        buffer
    );
}

/*
 * Draw the list of options and their current values in the top left corner.
 */
static void _um_tui_classic_draw_options(int zoom,
                                            const ClassicOptions *options) {
    int start_y = 2,
        start_x = 0;
    mvaddstr(start_y,     start_x,     TEXT_OPTIONS_TITLE);
    attr_off(A_BOLD, NULL);
    mvaddstr(start_y + 2, start_x + 9, TEXT_OPTIONS_WINMODE);
    mvaddstr(start_y + 4, start_x + 9, TEXT_OPTIONS_COVERONLOSS);
    mvaddstr(start_y + 5, start_x + 9, TEXT_OPTIONS_NOEXPANDZEROTILES);
    mvaddstr(start_y + 6, start_x + 9, TEXT_OPTIONS_ZOOM);
    attr_on(COLOR_PAIR(6), NULL);
    mvaddstr(start_y + 2, start_x + 1,
                TEXT_OPTIONS_WINMODE_VALUES[options->win_mode]);
    mvaddstr(start_y + 4, start_x + 1,
                TEXT_OPTIONS_VALUES[options->cover_on_loss]);
    mvaddstr(start_y + 5, start_x + 1,
                TEXT_OPTIONS_VALUES[options->no_expand_zero_tiles]);
    mvaddstr(start_y + 6, start_x + 1, TEXT_OPTIONS_VALUES[zoom]);
    attr_off(COLOR_PAIR(6), NULL);
    attr_on(A_BOLD, NULL);
}

/*
 * Draw the game state display above the board.
 */
static void _um_tui_classic_draw_state(ClassicState state, void *data) {
    CallbackData *cb_data = (CallbackData *) data;
    mvaddstr(
        UM_TUI_BOARD_Y(cb_data->params.height, cb_data->zoom) - 2,
        UM_TUI_CENTER(COLS, strlen(TEXT_STATE_CLEAR)),
        TEXT_STATE_CLEAR
    );
    mvaddstr(
        UM_TUI_BOARD_Y(cb_data->params.height, cb_data->zoom) - 2,
        UM_TUI_CENTER(COLS, strlen(TEXT_STATES[state])),
        TEXT_STATES[state]
    );
}

/*
 * Draw the game timer value above the top right corner of the board.
 */
static void _um_tui_classic_draw_timer(double seconds, int board_width,
                                        int board_height, int zoom) {
    char buffer[] = "...";
    if (seconds < 1000) {
        sprintf(buffer, "%3.f", seconds);
    }
    mvaddstr(
        UM_TUI_BOARD_Y(board_height, zoom) - 2,
        UM_TUI_BOARD_X(board_width, zoom) +
            UM_TUI_BOARD_DIM(board_width, zoom) - 3,
        buffer
    );
}

/*
 * Draw the labels on the WinMode selection screen.
 */
static void _um_tui_classic_winmode_draw(void) {
    int x_start = (COLS - strlen(TEXT_WINMODE_REVEAL) + 4) / 2,
        y_start = UM_TUI_CENTER(LINES, 9);
    mvaddstr(y_start,     x_start,     TEXT_WINMODE_PROMPT);
    mvaddstr(y_start + 2, x_start + 4, TEXT_WINMODE_REVEAL);
    mvaddstr(y_start + 4, x_start + 4, TEXT_WINMODE_FLAGS);
    mvaddstr(y_start + 6, x_start + 4, TEXT_WINMODE_EITHER);
    mvaddstr(y_start + 8, x_start + 4, TEXT_WINMODE_BOTH);
}

/*
 * Draw the cursor on the WinMode selection screen.
 */
static void _um_tui_classic_winmode_draw_selection(ClassicWinMode selection,
                                                    int clear_old) {
    int x = (COLS - strlen(TEXT_WINMODE_REVEAL) + 4) / 2 + 2,
        y_start = UM_TUI_CENTER(LINES, 9) + 2;
    if (clear_old) {
        mvaddch(y_start,     x, ' ');
        mvaddch(y_start + 2, x, ' ');
        mvaddch(y_start + 4, x, ' ');
        mvaddch(y_start + 6, x, ' ');
    }
    mvaddch(y_start + 2 * selection, x, '>');
}

/*
 * Display a dialog allowing the user to change the WinMode. Returns 1 if the
 * WinMode is changed.
 */
static ClassicWinMode _um_tui_classic_winmode(ClassicWinMode current) {
    int input = 0,
        selection = current;
    do {
        int redraw_all = um_tui_is_size_changed(),
            redraw_selection = 0;
        switch(input) {
            case 0:
                redraw_all = 1;
                break;
            case KEY_UP:
                --selection;
                redraw_selection = 1;
                break;
            case KEY_DOWN:
                ++selection;
                redraw_selection = 1;
                break;
            case 10:
            case 13:
            case KEY_B2:
                return selection;
        }
        if (redraw_selection) {
            selection = um_tui_positive_modulo(
                selection, NUM_SELECTIONS_WINMODE
            );
        }
        if (redraw_all) {
            um_tui_draw_meta(
                TEXT_TITLE,
                TEXT_WINMODE_TITLE,
                TEXT_WINMODE_DIRECTIONS,
                TEXT_WINMODE_DIRECTIONS_LENGTH
            );
            _um_tui_classic_winmode_draw();
            _um_tui_classic_winmode_draw_selection(selection, 0);
        } else if (redraw_selection) {
            _um_tui_classic_winmode_draw_selection(selection, 1);
        }
        if (redraw_selection || redraw_all) {
            refresh();
        }
        input = getch();
    } while (!um_tui_is_quit_key(input));
    return current;
}

/*
 * Public functions.
 */

/*
 * Classic game.
 */
int um_tui_classic(config_setting_t *config) {
    int input = 0;
    double last_time = 0;
    CallbackData data = {
        {0, 0},
        {0, 0, 0},
        1
    };
    ClassicCallbacks callbacks = {
        _um_tui_classic_draw_board,
        _um_tui_classic_draw_flag_count,
        _um_tui_classic_draw_state,
        _um_tui_classic_draw_tile,
        NULL
    };
    ClassicOptions options = {0, 0, 1, EITHER};
    ClassicGame *game;
    ClassicWinMode new_win_mode = EITHER;
    Coords new_selection = {0, 0};
    Tile **tiles;
    data.params = UM_CLASSIC_DEFAULT;
    _um_tui_classic_config_load(config, &options, &data.params, &data.zoom);
    callbacks.data = &data;
    game = um_classic_create(&callbacks, &options, &data.params);
    if (!game) {
        return 1;
    }
    tiles = um_classic_tiles(game);
    do {
        int check_selection = 0,
            redraw_all = um_tui_is_size_changed(),
            redraw_options = 0,
            redraw_selection = 0,
            redraw_timer = 0;
        double game_time;
        Coords old_selection = {0, 0};
        switch(input) {
            case 0:
                redraw_all = 1;
                break;
            case ' ':
                um_classic_reveal(game, &data.selection);
                break;
            case 'c':
            case 'C':
                options.cover_on_loss = !options.cover_on_loss;
                um_classic_options(game, &options);
                redraw_options = 1;
                break;
            case 'e':
            case 'E':
                um_classic_question(game, &data.selection);
                break;
            case 'f':
            case 'F':
                um_classic_flag(game, &data.selection);
                break;
            case 'm':
            case 'M':
                if (um_tui_board_params(TEXT_TITLE, &data.params.width,
                                        &UM_CLASSIC_MIN.width, 2)) {
                    tiles = um_classic_parameters(game, &data.params);
                    if (!tiles) {
                        return 1;
                    }
                    check_selection = 1;
                }
                redraw_all = 1;
                break;
            case 'r':
            case 'R':
                um_classic_reset(game);
                break;
            case 's':
            case 'S':
                if (um_tui_board_params(TEXT_TITLE, &data.params.width,
                                        &UM_CLASSIC_MIN.width, 0)) {
                    tiles = um_classic_parameters(game, &data.params);
                    if (!tiles) {
                        return 1;
                    }
                    check_selection = 1;
                }
                redraw_all = 1;
                break;
            case 'v':
            case 'V':
                um_classic_reveal_around(game, &data.selection);
                break;
            case 'w':
            case 'W':
                new_win_mode = _um_tui_classic_winmode(options.win_mode);
                if (new_win_mode != options.win_mode) {
                    options.win_mode = new_win_mode;
                    um_classic_options(game, &options);
                }
                redraw_all = 1;
                break;
            case 'x':
            case 'X':
                options.no_expand_zero_tiles = !options.no_expand_zero_tiles;
                um_classic_options(game, &options);
                redraw_options = 1;
                break;
            case 'z':
            case 'Z':
                data.zoom = !data.zoom;
                redraw_all = 1;
                break;
            case KEY_UP:
                if (data.selection.y > 0) {
                    --new_selection.y;
                }
                break;
            case KEY_DOWN:
                if (data.selection.y < data.params.height - 1) {
                    ++new_selection.y;
                }
                break;
            case KEY_LEFT:
                if (data.selection.x > 0) {
                    --new_selection.x;
                }
                break;
            case KEY_RIGHT:
                if (data.selection.x < data.params.width - 1) {
                    ++new_selection.x;
                }
                break;
        }
        if (check_selection) {
            if (data.selection.x >= data.params.width) {
                data.selection.x = data.params.width - 1;
                new_selection.x = data.selection.x;
            }
            if (data.selection.y >= data.params.height) {
                data.selection.y = data.params.height - 1;
                new_selection.y = data.selection.y;
            }
        }
        if (data.selection.x != new_selection.x ||
            data.selection.y != new_selection.y) {
            old_selection.x = data.selection.x;
            old_selection.y = data.selection.y;
            data.selection.x = new_selection.x;
            data.selection.y = new_selection.y;
            redraw_selection = 1;
        }
        game_time = um_classic_time(game);
        if (game_time != last_time) {
            redraw_timer = 1;
            last_time = game_time;
        }
        if (redraw_all) {
            um_tui_draw_meta(
                TEXT_TITLE,
                NULL,
                TEXT_DIRECTIONS,
                TEXT_DIRECTIONS_LENGTH
            );
            um_classic_update(game);
            redraw_options = 1;
            redraw_timer = 1;
        } else if (redraw_selection) {
            _um_tui_classic_draw_tile(
                &tiles[old_selection.x][old_selection.y], (void *) &data
            );
            _um_tui_classic_draw_tile(
                &tiles[data.selection.x][data.selection.y], (void *) &data
            );
        }
        if (redraw_options) {
            _um_tui_classic_draw_options(data.zoom, &options);
        }
        if (redraw_timer) {
            _um_tui_classic_draw_timer(game_time, data.params.width,
                                        data.params.height, data.zoom);
        }
        if (redraw_timer || redraw_options || redraw_selection || redraw_all) {
            refresh();
        }
        input = getch();
    } while (!um_tui_is_quit_key(input));
    um_classic_destroy(game);
    _um_tui_classic_config_save(config, &options, &data.params, data.zoom);
    return 0;
}
