/*
 * unmine/tui_common.c
 *
 * Copyright 2017 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#define _POSIX_C_SOURCE 199506L
#define _XOPEN_SOURCE 1
#define _XOPEN_SOURCE_EXTENDED 1

#include "tui_common.h"

#include <string.h>

#include <ncursesw/curses.h>

#include "unmine.h"

/*
 * Escape key constant. (This should be defined by ncurses!)
 */
static const int KEY_ESCAPE = 27;

/*
 * Menu item counts.
 */
static const int NUM_SELECTIONS_BOARD_PARAMS = 3;

/*
 * Strings.
 */
static const char *CONFIG_NAME_ZOOM = "zoom";
static const char * const TEXT_BOARD_PARAMS_DIRECTIONS[] = {
    "[up/down]    move selection",
    "[left/right] increase/decrease",
    "[enter]      ok",
    "[q/esc]      cancel"
};
static const int TEXT_BOARD_PARAMS_DIRECTIONS_LENGTH = 4;
static const char *TEXT_BOARD_PARAMS_TITLE = "Change # Mines / Size";
static const char *TEXT_BOARD_PARAMS_HEIGHT = "Height";
static const char *TEXT_BOARD_PARAMS_MINES = "Mines";
static const char *TEXT_BOARD_PARAMS_WIDTH = "Width";
static const char *TEXT_TITLE_SEPARATOR = " > ";

/*
 * Private functions.
 */

/*
 * Draw the labels on the board parameters dialog.
 */
static void _um_tui_board_params_draw(void) {
    int x = UM_TUI_CENTER(COLS, strlen(TEXT_BOARD_PARAMS_HEIGHT) + 11),
        y_start = UM_TUI_CENTER(LINES, 5);
    mvaddstr(y_start,     x, TEXT_BOARD_PARAMS_WIDTH);
    mvaddstr(y_start + 2, x, TEXT_BOARD_PARAMS_HEIGHT);
    mvaddstr(y_start + 4, x, TEXT_BOARD_PARAMS_MINES);
}

/*
 * Draw the values on the board parameters dialog.
 */
static void _um_tui_board_params_draw_selection(int values[3],
                                                int selection) {
    int i,
        x = UM_TUI_CENTER(COLS, strlen(TEXT_BOARD_PARAMS_HEIGHT) + 11) +
            strlen(TEXT_BOARD_PARAMS_HEIGHT) + 2,
        y_start = UM_TUI_CENTER(LINES, 5);
    for (i = 0; i < 3; ++i) {
        char buffer[] = "...";
        mvaddch(y_start + 2*i, x, ' ');
        addch(' ');
        if (i == selection) {
            addch('-');
        } else {
            addch(' ');
        }
        addch(' ');
        if (values[i] < 1000) {
            sprintf(buffer, "%3d", values[i]);
        }
        addstr(buffer);
        addch(' ');
        if (i == selection) {
            addch('+');
        } else {
            addch(' ');
        }
    }
}

/*
 * Public functions.
 */

/*
 * Display a dialog allowing the user to change board parameters. Returns 1 if
 * any parameter is changed.
 */
int um_tui_board_params(const char *title, int params[3], const int mins[3],
                        int selection) {
    int input = 0,
        new_params[3];
    new_params[0] = params[0];
    new_params[1] = params[1];
    new_params[2] = params[2];
    do {
        int max,
            modifier = 0,
            redraw_all = um_tui_is_size_changed(),
            redraw_selection = 0;
        switch(input) {
            case 0:
                redraw_all = 1;
                break;
            case KEY_UP:
                --selection;
                redraw_selection = 1;
                break;
            case KEY_DOWN:
                ++selection;
                redraw_selection = 1;
                break;
            case KEY_LEFT:
                if (new_params[selection] > mins[selection]) {
                    --modifier;
                }
                break;
            case KEY_RIGHT:
                if (!selection) {
                    max = COLS - 2;
                } else if (selection == 1) {
                    max = LINES - 3;
                } else {
                    max = new_params[0] * new_params[1] - 2;
                }
                if (new_params[selection] < max) {
                    ++modifier;
                }
                break;
            case 10:
            case 13:
            case KEY_B2:
                if (new_params[0] != params[0] ||
                    new_params[1] != params[1] ||
                    new_params[2] != params[2]) {
                    params[0] = new_params[0];
                    params[1] = new_params[1];
                    params[2] = new_params[2];
                    return 1;
                }
                return 0;
        }
        if (redraw_selection) {
            selection = um_tui_positive_modulo(
                selection,
                NUM_SELECTIONS_BOARD_PARAMS
            );
        }
        if (modifier) {
            new_params[selection] += modifier;
            if (modifier < 0 && selection < 2 &&
                new_params[2] > new_params[0] * new_params[1] - 2) {
                new_params[2] = new_params[0] * new_params[1] - 2;
            }
            redraw_selection = 1;
        }
        if (redraw_all) {
            um_tui_draw_meta(
                title,
                TEXT_BOARD_PARAMS_TITLE,
                TEXT_BOARD_PARAMS_DIRECTIONS,
                TEXT_BOARD_PARAMS_DIRECTIONS_LENGTH
            );
            _um_tui_board_params_draw();
            redraw_selection = 1;
        }
        if (redraw_selection) {
            _um_tui_board_params_draw_selection(new_params, selection);
            refresh();
        }
        input = getch();
    } while (!um_tui_is_quit_key(input));
    return 0;
}

/*
 * Load zoom from config.
 */
void um_tui_config_load_zoom(config_setting_t *config, int *zoom) {
    if (config_setting_get_member(config, CONFIG_NAME_ZOOM)) {
        config_setting_lookup_bool(config, CONFIG_NAME_ZOOM, zoom);
    } else {
        config_setting_add(config, CONFIG_NAME_ZOOM, CONFIG_TYPE_BOOL);
    }
}

/*
 * Save zoom to config.
 */
void um_tui_config_save_zoom(config_setting_t *config, int zoom) {
    config_setting_t *setting_zoom;
    setting_zoom = config_setting_get_member(config, CONFIG_NAME_ZOOM);
    if (setting_zoom) {
        config_setting_set_bool(setting_zoom, zoom);
    }
}

/*
 * Draw the program name, version, and (optionally) a title and subtitle for
 * the current screen in the top left corner, and (optionally) help/directions
 * in the bottom left corner.
 */
void um_tui_draw_meta(const char *title, const char *subtitle,
                        const char * const *directions,
                        int directions_length) {
    clear();
    attr_off(A_BOLD, NULL);
    mvaddstr(0, 0, UM_NAME);
    addch(' ');
    addstr(UM_VERSION);
    if (title) {
        addstr(TEXT_TITLE_SEPARATOR);
        addstr(title);
    }
    if (subtitle) {
        addstr(TEXT_TITLE_SEPARATOR);
        addstr(subtitle);
    }
    if (directions) {
        int i,
            start_y = LINES - directions_length;
        for (i = 0; i < directions_length; ++i) {
            mvaddstr(start_y + i, 0, directions[i]);
        }
    }
    attr_on(A_BOLD, NULL);
}

/*
 * Returns 1 if the given key should quit the current screen.
 */
int um_tui_is_quit_key(int key) {
    return key == KEY_ESCAPE || key == 'q' || key == 'Q';
}

/*
 * Returns 1 if the window size has changed since the last time this was called
 * (returns 1 the first time it's called).
 */
int um_tui_is_size_changed(void) {
    static int width = 0,
               height = 0;
    int retval = (width != COLS || height != LINES);
    width = COLS;
    height = LINES;
    return retval;
}

/*
 * Returns positive i mod n.
 */
int um_tui_positive_modulo(int i, int n) {
    return (i % n + n) % n;
}
