/*
 * unmine/tui_flags.c
 *
 * Copyright 2018 Kyle Stevenson <stevensonkd@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#define _POSIX_C_SOURCE 199506L
#define _XOPEN_SOURCE 1
#define _XOPEN_SOURCE_EXTENDED 1

#include "tui_flags.h"

#include <string.h>

#include <ncursesw/curses.h>
#include <pthread.h>
#ifdef __MINGW32__
    #include <windows.h>
#else /* __MINGW32__ */
    #include <sys/time.h>
    #include <sys/types.h>
    #include <unistd.h>
#endif /* __MINGW32__ */

#include "coords.h"
#include "flags.h"
#include "flags_ai.h"
#include "flags_config.h"
#include "tile.h"
#include "tui_common.h"

/*
 * Constants.
 */
/* Colours are defined for only 7 players. */
#define _UM_TUI_FLAGS_MAX_PLAYERS 7

/*
 * Strings.
 */
static const char *TEXT_AI_LEVELS_CLEAR = "            ";
static const char * const TEXT_DIRECTIONS[] = {
    "[arrows] move selection",
    "[space]  select",
    "[b]      toggle bomb",
    "[r]      reset",
    "[m/s]    change # mines / size",
    "[p]      change players",
    "[q/esc]  quit"
};
static const int TEXT_DIRECTIONS_LENGTH = 7;
static const char *TEXT_OPTIONS_TITLE = "OPTIONS";
static const char *TEXT_OPTIONS_ALLOWTIES = "[a]llow ties";
static const char *TEXT_OPTIONS_BOMBENDSTURN = "b[o]mb ends turn";
static const char *TEXT_OPTIONS_BOMBRANGE = "bomb range [h-/j+]";
static const char *TEXT_OPTIONS_COVERONDONE = "[c]over on done";
static const char *TEXT_OPTIONS_NOEXPANDZEROTILES = "no e[x]pand zero tiles";
static const char *TEXT_OPTIONS_NUMBOMBS = "# bombs [y-/u+]";
static const char *TEXT_OPTIONS_ZOOM = "[z]oom";
static const char * const TEXT_OPTIONS_VALUES[] = {"off", "on "};
static const char *TEXT_PLAYERS_AI = "AI #";
static const char *TEXT_PLAYERS_AI_CLEAR = "       ";
static const char * const TEXT_PLAYERS_DIRECTIONS[] = {
    "[up/down]    move selection",
    "[left/right] increase/decrease/change",
    "[enter]      ok",
    "[q/esc]      cancel"
};
static const int TEXT_PLAYERS_DIRECTIONS_LENGTH = 4;
static const char *TEXT_PLAYERS_HEADER1 = "PLAYERS    ";
static const char *TEXT_PLAYERS_HEADER2 = "bombs score";
static const char *TEXT_PLAYERS_NUM_AI = "# AI Players";
static const char *TEXT_PLAYERS_NUM_HOTSEAT = "# Hotseat Players";
static const char *TEXT_PLAYERS_PLAYER = "Hotseat #";
static const char *TEXT_PLAYERS_TITLE = "Change Players";
static const char *TEXT_STATE_CLEAR = "       ";
static const char * const TEXT_STATES[] = {
    "READY", "GO!", "WINNER!"
};
static const char *TEXT_TITLE = "Flags";

/*
 * Types.
 */

/* State shared between the main loop and callbacks, and, equivalently, between
 * the main and AI threads. All members are protected by the mutex. Each member
 * is read by both threads, but written by only one, indicated below. Since a
 * writer thread knows its data won't change, it need not acquire the mutex to
 * read it. However, this optimization can't be used everywhere because the
 * mutex is also used to synchronize ncurses interactions. Using a separate
 * mutex for ncurses is possible, but probably not worth the complexity.
 */
typedef struct {
    pthread_mutex_t mutex;
    FlagsParameters params;     /* main loop */
    FlagsOptions options;       /* main loop */
    Coords selection;           /* main loop */
    Tile **tiles;               /* callbacks */
    const FlagsPlayer *players; /* callbacks */
    int bomb,                   /* main loop */
        turn_player,            /* callbacks */
        zoom;                   /* main loop */
} CallbackData;

/*
 * Private functions.
 */

static int  _um_tui_flags_check_selected(const Tile *tile,
                                            const Coords *centre, int bomb,
                                            int range);
static void _um_tui_flags_config_load(config_setting_t *config,
                                        FlagsOptions *options,
                                        FlagsParameters *params, int *zoom);
static void _um_tui_flags_config_save(config_setting_t *config,
                                        FlagsOptions *options,
                                        FlagsParameters *params, int zoom);
static void _um_tui_flags_draw_board(Tile **tiles, const CallbackData *data);
static void _um_tui_flags_draw_mine_count(int mine_count,
                                            const CallbackData *data);
static void _um_tui_flags_draw_options(const FlagsOptions *options, int zoom);
static void _um_tui_flags_draw_players(const FlagsPlayer *players,
                                        int num_players);
static void _um_tui_flags_draw_selection(const Coords *centre, int range,
                                            const CallbackData *data);
static void _um_tui_flags_draw_state(FlagsState state, int height, int zoom);
static void _um_tui_flags_draw_tile(const Tile *tile,
                                    const CallbackData *data);
static void _um_tui_flags_draw_turn(int turn_player, int num_players);
static void _um_tui_flags_draw_win(const FlagsPlayer *players,
                                    int num_players);
static void _um_tui_flags_players_draw_labels(int num_ai);
static void _um_tui_flags_players_draw_values(int num_hotseat, int num_ai,
                                                const FlagsAILevel *ai_levels,
                                                int selection);
static int  _um_tui_flags_players(FlagsParameters *params);
static int  _um_tui_flags_run(FlagsGame *game, CallbackData *data);
static void _um_tui_flags_update_board(Tile **tiles, void *data);
static void _um_tui_flags_update_mine_count(int mine_count, void *data);
static void _um_tui_flags_update_players(const FlagsPlayer *players,
                                            int num_players, void *data);
static void _um_tui_flags_update_state(FlagsState state, void *data);
static void _um_tui_flags_update_tile(const Tile *tile, void *data);
static void _um_tui_flags_update_turn(int player, void *data);
static int  _um_tui_flags_wait(void);

/*
 * Returns 1 if the given tile is selected.
 */
static int _um_tui_flags_check_selected(const Tile *tile, const Coords *centre,
                                        int bomb, int range) {
    if (!bomb) {
        range = 0;
    }
    return tile->x >= centre->x - range && tile->x <= centre->x + range &&
           tile->y >= centre->y - range && tile->y <= centre->y + range;
}

/*
 * Load configuration.
 */
static void _um_tui_flags_config_load(config_setting_t *config,
                                        FlagsOptions *options,
                                        FlagsParameters *params, int *zoom) {
    um_flags_config_load(config, options, params, _UM_TUI_FLAGS_MAX_PLAYERS);
    um_tui_config_load_zoom(config, zoom);
}

/*
 * Save configuration.
 */
static void _um_tui_flags_config_save(config_setting_t *config,
                                        FlagsOptions *options,
                                        FlagsParameters *params, int zoom) {
    um_flags_config_save(config, options, params, _UM_TUI_FLAGS_MAX_PLAYERS);
    um_tui_config_save_zoom(config, zoom);
}

/*
 * Draw the whole board (wraps _um_tui_flags_draw_tile).
 */
static void _um_tui_flags_draw_board(Tile **tiles, const CallbackData *data) {
    int i, j;
    for (i = 0; i < data->params.width; ++i) {
        for (j = 0; j < data->params.height; ++j) {
            _um_tui_flags_draw_tile(&tiles[i][j], data);
        }
    }
}

/*
 * Draw the mine count.
 */
static void _um_tui_flags_draw_mine_count(int mine_count,
                                            const CallbackData *data) {
    char buffer[] = "...";
    if (mine_count < 1000) {
        sprintf(buffer, "%3d", mine_count);
    }
    mvaddstr(
        UM_TUI_BOARD_Y(data->params.height, data->zoom) - 2,
        UM_TUI_BOARD_X(data->params.width, data->zoom),
        buffer
    );
}

/*
 * Draw the list of options and their current values in the top left corner.
 */
static void _um_tui_flags_draw_options(const FlagsOptions *options, int zoom) {
    char num_buffer[] = "...",
         range_buffer[] = "...";
    int start_y = 2,
        start_x = 0;
    if (options->num_bombs < 1000) {
        sprintf(num_buffer, "%3d", options->num_bombs);
    }
    if (options->bomb_range < 1000) {
        sprintf(range_buffer, "%3d", options->bomb_range);
    }
    mvaddstr(start_y,     start_x,     TEXT_OPTIONS_TITLE);
    attr_off(A_BOLD, NULL);
    mvaddstr(start_y + 2, start_x + 6, TEXT_OPTIONS_NUMBOMBS);
    mvaddstr(start_y + 3, start_x + 6, TEXT_OPTIONS_BOMBRANGE);
    mvaddstr(start_y + 5, start_x + 6, TEXT_OPTIONS_ALLOWTIES);
    mvaddstr(start_y + 6, start_x + 6, TEXT_OPTIONS_BOMBENDSTURN);
    mvaddstr(start_y + 7, start_x + 6, TEXT_OPTIONS_COVERONDONE);
    mvaddstr(start_y + 8, start_x + 6, TEXT_OPTIONS_NOEXPANDZEROTILES);
    mvaddstr(start_y + 9, start_x + 6, TEXT_OPTIONS_ZOOM);
    attr_on(COLOR_PAIR(6), NULL);
    mvaddstr(start_y + 2, start_x + 1, num_buffer);
    mvaddstr(start_y + 3, start_x + 1, range_buffer);
    mvaddstr(start_y + 5, start_x + 1,
                TEXT_OPTIONS_VALUES[options->allow_ties]);
    mvaddstr(start_y + 6, start_x + 1,
                TEXT_OPTIONS_VALUES[options->bomb_ends_turn]);
    mvaddstr(start_y + 7, start_x + 1,
                TEXT_OPTIONS_VALUES[options->cover_on_done]);
    mvaddstr(start_y + 8, start_x + 1,
                TEXT_OPTIONS_VALUES[options->no_expand_zero_tiles]);
    mvaddstr(start_y + 9, start_x + 1, TEXT_OPTIONS_VALUES[zoom]);
    attr_off(COLOR_PAIR(6), NULL);
    attr_on(A_BOLD, NULL);
}

/*
 * Draw the list of players, and their bomb counts and scores, in the top right
 * corner.
 */
static void _um_tui_flags_draw_players(const FlagsPlayer *players,
                                        int num_players) {
    int h1_length = strlen(TEXT_PLAYERS_HEADER1),
        num_ai = 0,
        num_hotseat = 0,
        x_start = COLS - h1_length - strlen(TEXT_PLAYERS_HEADER2) - 1,
        y_start = 2,
        i;
    mvaddstr(y_start, x_start, TEXT_PLAYERS_HEADER1);
    attr_off(A_BOLD, NULL);
    addstr(TEXT_PLAYERS_HEADER2);
    for (i = 0; i < num_players; ++i) {
        char buffer_bombs[] = ".....",
             buffer_score[] = ".....";
        if (players[i].num_bombs < 100000) {
            sprintf(buffer_bombs, "%5d", players[i].num_bombs);
        }
        if (players[i].score < 100000) {
            sprintf(buffer_score, "%5d", players[i].score);
        }
        attr_on(COLOR_PAIR(31 + i), NULL);
        attr_on(A_BOLD, NULL);
        if (players[i].ai > NO_AI) {
            ++num_ai;
            mvaddstr(y_start + 1 + i, x_start, TEXT_PLAYERS_AI);
            addch(num_ai + '0');
        } else {
            ++num_hotseat;
            mvaddstr(y_start + 1 + i, x_start, TEXT_PLAYERS_PLAYER);
            addch(num_hotseat + '0');
        }
        attr_off(COLOR_PAIR(31 + i), NULL);
        attr_off(A_BOLD, NULL);
        mvaddstr(y_start + 1 + i, x_start + h1_length,     buffer_bombs);
        mvaddstr(y_start + 1 + i, x_start + h1_length + 6, buffer_score);
    }
    attr_on(A_BOLD, NULL);
}

/*
 * Draw the tiles in the square centred at centre and extending range tiles in
 * each direction.
 */
static void _um_tui_flags_draw_selection(const Coords *centre, int range,
                                            const CallbackData *data) {
    int i_start = centre->x - range,
        j_start = centre->y - range,
        max_x = data->params.width,
        max_y = data->params.height,
        i, j;
    if (i_start < 0) {
        i_start = 0;
    }
    if (j_start < 0) {
        j_start = 0;
    }
    for (i = i_start; i <= centre->x + range && i < max_x; ++i) {
        for (j = j_start; j <= centre->y + range && j < max_y; ++j) {
            _um_tui_flags_draw_tile(&data->tiles[i][j], data);
        }
    }
}

/*
 * Draw the current state, centered above the board.
 */
static void _um_tui_flags_draw_state(FlagsState state, int height, int zoom) {
    int y = UM_TUI_BOARD_Y(height, zoom) - 2;
    mvaddstr(y, UM_TUI_CENTER(COLS, strlen(TEXT_STATE_CLEAR)),
                TEXT_STATE_CLEAR);
    mvaddstr(y, UM_TUI_CENTER(COLS, strlen(TEXT_STATES[state])),
                TEXT_STATES[state]);
}

/*
 * Draw the given Tile.
 */
static void _um_tui_flags_draw_tile(const Tile *tile,
                                    const CallbackData *data) {
    int pair_index = 1,
        selected = _um_tui_flags_check_selected(tile, &data->selection,
                                        data->bomb, data->options.bomb_range),
        x = UM_TUI_BOARD_X(data->params.width, data->zoom) +
                (data->zoom + 1) * tile->x,
        y = UM_TUI_BOARD_Y(data->params.height, data->zoom) +
                (data->zoom + 1) * tile->y;
    chtype face = ' ';
    if (tile->revealed) {
        if (tile->mined) {
            face = '*';
            pair_index = 23 + selected;
        } else {
            if (tile->count) {
                face = tile->count + '0';
            }
            pair_index = tile->count + 1 + selected * 9;
        }
    } else {
        if (tile->face >= PLAYER_FLAG) {
            face = '>';
            pair_index = 31 + tile->face - PLAYER_FLAG + selected * 7;
        } else {
            pair_index = 29 + selected;
        }
    }
    attr_on(COLOR_PAIR(pair_index), NULL);
    mvaddch(y, x, face);
    attr_off(COLOR_PAIR(pair_index), NULL);
}

/*
 * Draw the turn indicator on the list of players.
 */
static void _um_tui_flags_draw_turn(int turn_player, int num_players) {
    int x = COLS - strlen(TEXT_PLAYERS_HEADER1) -
                strlen(TEXT_PLAYERS_HEADER2) - 3,
        y_start = 3,
        i;
    for (i = 0; i < num_players; ++i) {
        chtype c = ' ';
        if (i == turn_player) {
            c = '>';
        }
        mvaddch(y_start + i, x, c);
    }
}

/*
 * Draw the win indicator(s) on the list of players.
 * TODO: draw winner's name(s) in the state area.
 */
static void _um_tui_flags_draw_win(const FlagsPlayer *players,
                                    int num_players) {
    int max = 0,
        x = COLS - strlen(TEXT_PLAYERS_HEADER1) -
                strlen(TEXT_PLAYERS_HEADER2) - 3,
        y_start = 3,
        i;
    for (i = 0; i < num_players; ++i) {
        if (players[i].score > max) {
            max = players[i].score;
        }
    }
    for (i = 0; i < num_players; ++i) {
        chtype c = ' ';
        if (players[i].score == max) {
            c = '*';
        }
        attr_on(COLOR_PAIR(31 + i), NULL);
        mvaddch(y_start + i, x, c);
        attr_off(COLOR_PAIR(31 + i), NULL);
    }
}

/*
 * Draw the labels on the board parameters dialog.
 */
static void _um_tui_flags_players_draw_labels(int num_ai) {
    int x = UM_TUI_CENTER(COLS, strlen(TEXT_PLAYERS_NUM_HOTSEAT) + 8),
        y_start = UM_TUI_CENTER(LINES, 4 + _UM_TUI_FLAGS_MAX_PLAYERS),
        i;
    mvaddstr(y_start,     x, TEXT_PLAYERS_NUM_HOTSEAT);
    mvaddstr(y_start + 2, x, TEXT_PLAYERS_NUM_AI);
    for (i = 0; i < _UM_TUI_FLAGS_MAX_PLAYERS; ++i) {
        if (i < num_ai) {
            char buffer[] = "...";
            if (i < 1000) {
                sprintf(buffer, "%d", i + 1);
            }
            mvaddstr(y_start + 4 + i, x, TEXT_PLAYERS_AI);
            addstr(buffer);
        } else {
            mvaddstr(y_start + 4 + i, x, TEXT_PLAYERS_AI_CLEAR);
        }
    }
}

/*
 * Draw the values on the player parameters dialog.
 */
static void _um_tui_flags_players_draw_values(int num_hotseat, int num_ai,
                                                const FlagsAILevel *ai_levels,
                                                int selection) {
    int i,
        len = strlen(TEXT_PLAYERS_NUM_HOTSEAT),
        x = UM_TUI_CENTER(COLS, len + 8) + len + 2,
        y_start = UM_TUI_CENTER(LINES, 4 + _UM_TUI_FLAGS_MAX_PLAYERS);
    for (i = 0; i < 2; ++i) {
        int val = num_hotseat;
        if (i) {
            val = num_ai;
        }
        mvaddch(y_start + 2*i, x, ' ');
        if (i == selection) {
            addch('-');
        } else {
            addch(' ');
        }
        addch(' ');
        addch(val + '0');
        addch(' ');
        if (i == selection) {
            addch('+');
        } else {
            addch(' ');
        }
    }
    for (i = 0; i < _UM_TUI_FLAGS_MAX_PLAYERS; ++i) {
        mvaddch(y_start + 4 + i, x, ' ');
        if (i + 2 == selection) {
            addch('<');
        } else {
            addch(' ');
        }
        addch(' ');
        if (i < num_ai) {
            int space = strlen(TEXT_AI_LEVELS_CLEAR) -
                        strlen(UM_FLAGS_AI_LEVEL_NAMES[ai_levels[i]]);
            addstr(UM_FLAGS_AI_LEVEL_NAMES[ai_levels[i]]);
            while (space > 0) {
                addch(' ');
                --space;
            }
        } else {
            addstr(TEXT_AI_LEVELS_CLEAR);
        }
        addch(' ');
        if (i + 2 == selection) {
            addch('>');
        } else {
            addch(' ');
        }
    }
}

/*
 * Display a dialog allowing the user to change the game's player parameters.
 * Returns 1 if any parameter is changed.
 */
static int _um_tui_flags_players(FlagsParameters *params) {
    int input = 0,
        new_ai = params->num_ai,
        new_hotseat = params->num_hotseat,
        selection = 0,
        i;
    FlagsAILevel new_levels[_UM_TUI_FLAGS_MAX_PLAYERS];
    for (i = 0; i < _UM_TUI_FLAGS_MAX_PLAYERS; ++i) {
        new_levels[i] = params->ai_levels[i];
    }
    do {
        int changed = 0,
            redraw_all = um_tui_is_size_changed(),
            redraw_labels = 0,
            redraw_values = 0;
        switch(input) {
            case 0:
                redraw_all = 1;
                break;
            case KEY_UP:
                --selection;
                redraw_values = 1;
                break;
            case KEY_DOWN:
                ++selection;
                redraw_values = 1;
                break;
            case KEY_LEFT:
                if (!selection && new_hotseat > UM_FLAGS_MIN.num_hotseat) {
                    --new_hotseat;
                    if (new_hotseat + new_ai < 2) {
                        ++new_ai;
                        redraw_labels = 1;
                    }
                    redraw_values = 1;
                } else if (selection == 1 && new_ai > UM_FLAGS_MIN.num_ai) {
                    --new_ai;
                    if (new_hotseat + new_ai < 2) {
                        ++new_hotseat;
                    }
                    redraw_labels = 1;
                    redraw_values = 1;
                } else if (selection > 1 &&
                           new_levels[selection - 2] > NO_AI + 1)  {
                    --new_levels[selection - 2];
                    redraw_values = 1;
                }
                break;
            case KEY_RIGHT:
                if (!selection && new_hotseat < UM_FLAGS_MAX.num_hotseat &&
                    new_hotseat + new_ai < _UM_TUI_FLAGS_MAX_PLAYERS) {
                    ++new_hotseat;
                    redraw_values = 1;
                } else if (selection == 1 && new_ai < UM_FLAGS_MAX.num_ai &&
                           new_hotseat + new_ai < _UM_TUI_FLAGS_MAX_PLAYERS) {
                    ++new_ai;
                    redraw_labels = 1;
                    redraw_values = 1;
                } else if (selection > 1 &&
                           new_levels[selection - 2] < NUM_AI_LEVELS - 1) {
                    ++new_levels[selection - 2];
                    redraw_values = 1;
                }
                break;
            case 10:
            case 13:
            case KEY_B2:
                if (new_ai != params->num_ai) {
                    changed = 1;
                    params->num_ai = new_ai;
                }
                if (new_hotseat != params->num_hotseat) {
                    changed = 1;
                    params->num_hotseat = new_hotseat;
                }
                for (i = 0; i < new_ai; ++i) {
                    if (new_levels[i] != params->ai_levels[i]) {
                        changed = 1;
                        params->ai_levels[i] = new_levels[i];
                    }
                }
                return changed;
        }
        if (redraw_values) {
            selection = um_tui_positive_modulo(selection, 2 + new_ai);
        }
        if (redraw_all) {
            um_tui_draw_meta(
                TEXT_TITLE,
                TEXT_PLAYERS_TITLE,
                TEXT_PLAYERS_DIRECTIONS,
                TEXT_PLAYERS_DIRECTIONS_LENGTH
            );
            redraw_labels = 1;
            redraw_values = 1;
        }
        if (redraw_labels) {
            _um_tui_flags_players_draw_labels(new_ai);
        }
        if (redraw_values) {
            _um_tui_flags_players_draw_values(new_hotseat, new_ai, new_levels,
                                                selection);
        }
        if (redraw_values || redraw_labels || redraw_all) {
            refresh();
        }
        input = getch();
    } while (!um_tui_is_quit_key(input));
    return 0;
}

/*
 * Run the main loop. Returns 1 on error; otherwise 0.
 */
static int _um_tui_flags_run(FlagsGame *game, CallbackData *data) {
    int input = 0;
    do {
        int check_selection = 0,
            redraw_all = 0,
            redraw_options = 0,
            redraw_range = -1;
        Coords old_selection = {-1, -1};
        switch(input) {
            case 0:
                redraw_all = 1;
                break;
            case ' ':
                pthread_mutex_lock(&data->mutex);
                if (data->turn_player < data->params.num_hotseat) {
                    int player = data->turn_player;
                    if (data->bomb) {
                        pthread_mutex_unlock(&data->mutex);
                        um_flags_bomb(game, player, &data->selection);
                        pthread_mutex_lock(&data->mutex);
                        if (!data->players[data->turn_player].num_bombs) {
                            data->bomb = !data->bomb;
                            redraw_range = data->options.bomb_range;
                        }
                        pthread_mutex_unlock(&data->mutex);
                    } else {
                        pthread_mutex_unlock(&data->mutex);
                        um_flags_select(game, player, &data->selection);
                    }
                } else {
                    pthread_mutex_unlock(&data->mutex);
                }
                break;
            case 'a':
            case 'A':
                pthread_mutex_lock(&data->mutex);
                data->options.allow_ties = !data->options.allow_ties;
                pthread_mutex_unlock(&data->mutex);
                um_flags_options(game, 0, &data->options);
                redraw_options = 1;
                break;
            case 'b':
            case 'B':
                pthread_mutex_lock(&data->mutex);
                data->bomb = !data->bomb;
                pthread_mutex_unlock(&data->mutex);
                redraw_range = data->options.bomb_range;
                break;
            case 'c':
            case 'C':
                pthread_mutex_lock(&data->mutex);
                data->options.cover_on_done = !data->options.cover_on_done;
                pthread_mutex_unlock(&data->mutex);
                um_flags_options(game, 0, &data->options);
                redraw_options = 1;
                break;
            case 'h':
            case 'H':
                if (data->options.bomb_range > 1) {
                    if (data->bomb) {
                        redraw_range = data->options.bomb_range;
                    }
                    pthread_mutex_lock(&data->mutex);
                    --data->options.bomb_range;
                    pthread_mutex_unlock(&data->mutex);
                    um_flags_options(game, 0, &data->options);
                    redraw_options = 1;
                }
                break;
            case 'j':
            case 'J':
                pthread_mutex_lock(&data->mutex);
                ++data->options.bomb_range;
                pthread_mutex_unlock(&data->mutex);
                if (data->bomb) {
                    redraw_range = data->options.bomb_range;
                }
                um_flags_options(game, 0, &data->options);
                redraw_options = 1;
                break;
            case 'm':
            case 'M':
                pthread_mutex_lock(&data->mutex);
                if (um_tui_board_params(TEXT_TITLE, &data->params.width,
                                        &UM_FLAGS_MIN.width, 2)) {
                    pthread_mutex_unlock(&data->mutex);
                    if (um_flags_parameters(game, 0, &data->params) == 1) {
                        return 1;
                    }
                    check_selection = 1;
                } else {
                    pthread_mutex_unlock(&data->mutex);
                }
                redraw_all = 1;
                break;
            case 'o':
            case 'O':
                pthread_mutex_lock(&data->mutex);
                data->options.bomb_ends_turn = !data->options.bomb_ends_turn;
                pthread_mutex_unlock(&data->mutex);
                um_flags_options(game, 0, &data->options);
                redraw_options = 1;
                break;
            case 'p':
            case 'P':
                pthread_mutex_lock(&data->mutex);
                if (_um_tui_flags_players(&data->params)) {
                    pthread_mutex_unlock(&data->mutex);
                    if (um_flags_parameters(game, 0, &data->params) == 1) {
                        return 1;
                    }
                } else {
                    pthread_mutex_unlock(&data->mutex);
                }
                redraw_all = 1;
                break;
            case 'r':
            case 'R':
                um_flags_reset(game, 0);
                break;
            case 's':
            case 'S':
                pthread_mutex_lock(&data->mutex);
                if (um_tui_board_params(TEXT_TITLE, &data->params.width,
                                        &UM_FLAGS_MIN.width, 0)) {
                    pthread_mutex_unlock(&data->mutex);
                    if (um_flags_parameters(game, 0, &data->params) == 1) {
                        return 1;
                    }
                    check_selection = 1;
                } else {
                    pthread_mutex_unlock(&data->mutex);
                }
                redraw_all = 1;
                break;
            case 'u':
            case 'U':
                pthread_mutex_lock(&data->mutex);
                ++data->options.num_bombs;
                pthread_mutex_unlock(&data->mutex);
                um_flags_options(game, 0, &data->options);
                redraw_options = 1;
                break;
            case 'x':
            case 'X':
                pthread_mutex_lock(&data->mutex);
                data->options.no_expand_zero_tiles =
                                        !data->options.no_expand_zero_tiles;
                pthread_mutex_unlock(&data->mutex);
                um_flags_options(game, 0, &data->options);
                redraw_options = 1;
                break;
            case 'z':
            case 'Z':
                pthread_mutex_lock(&data->mutex);
                data->zoom = !data->zoom;
                pthread_mutex_unlock(&data->mutex);
                redraw_all = 1;
                break;
            case 'y':
            case 'Y':
                if (data->options.num_bombs) {
                    pthread_mutex_lock(&data->mutex);
                    --data->options.num_bombs;
                    pthread_mutex_unlock(&data->mutex);
                    um_flags_options(game, 0, &data->options);
                    redraw_options = 1;
                }
                break;
            case KEY_UP:
                if (data->selection.y > 0) {
                    old_selection = data->selection;
                    redraw_range = data->bomb ? data->options.bomb_range : 0;
                    pthread_mutex_lock(&data->mutex);
                    --data->selection.y;
                    pthread_mutex_unlock(&data->mutex);
                }
                break;
            case KEY_DOWN:
                if (data->selection.y < data->params.height - 1) {
                    old_selection = data->selection;
                    redraw_range = data->bomb ? data->options.bomb_range : 0;
                    pthread_mutex_lock(&data->mutex);
                    ++data->selection.y;
                    pthread_mutex_unlock(&data->mutex);
                }
                break;
            case KEY_LEFT:
                if (data->selection.x > 0) {
                    old_selection = data->selection;
                    redraw_range = data->bomb ? data->options.bomb_range : 0;
                    pthread_mutex_lock(&data->mutex);
                    --data->selection.x;
                    pthread_mutex_unlock(&data->mutex);
                }
                break;
            case KEY_RIGHT:
                if (data->selection.x < data->params.width - 1) {
                    old_selection = data->selection;
                    redraw_range = data->bomb ? data->options.bomb_range : 0;
                    pthread_mutex_lock(&data->mutex);
                    ++data->selection.x;
                    pthread_mutex_unlock(&data->mutex);
                }
                break;
        }
        if (!redraw_all) {
            pthread_mutex_lock(&data->mutex);
            redraw_all = um_tui_is_size_changed();
            pthread_mutex_unlock(&data->mutex);
        }
        if (check_selection) {
            if (data->selection.x >= data->params.width) {
                pthread_mutex_lock(&data->mutex);
                data->selection.x = data->params.width - 1;
                pthread_mutex_unlock(&data->mutex);
                redraw_range = data->bomb ? data->options.bomb_range : 0;
            }
            if (data->selection.y >= data->params.height) {
                pthread_mutex_lock(&data->mutex);
                data->selection.y = data->params.height - 1;
                pthread_mutex_unlock(&data->mutex);
                redraw_range = data->bomb ? data->options.bomb_range : 0;
            }
        }
        if (redraw_all) {
            pthread_mutex_lock(&data->mutex);
            um_tui_draw_meta(
                TEXT_TITLE,
                NULL,
                TEXT_DIRECTIONS,
                TEXT_DIRECTIONS_LENGTH
            );
            pthread_mutex_unlock(&data->mutex);
            um_flags_update(game);
            redraw_options = 1;
        } else if (redraw_range > -1) {
            pthread_mutex_lock(&data->mutex);
            if (old_selection.x > -1 && old_selection.y > -1) {
                _um_tui_flags_draw_selection(&old_selection, redraw_range,
                                                data);
            }
            _um_tui_flags_draw_selection(&data->selection, redraw_range, data);
            pthread_mutex_unlock(&data->mutex);
        }
        if (redraw_options) {
            pthread_mutex_lock(&data->mutex);
            _um_tui_flags_draw_options(&data->options, data->zoom);
            pthread_mutex_unlock(&data->mutex);
        }
        if (redraw_range > -1 || redraw_options || redraw_all) {
            pthread_mutex_lock(&data->mutex);
            refresh();
            pthread_mutex_unlock(&data->mutex);
        }
        if (_um_tui_flags_wait() > 0) {
            pthread_mutex_lock(&data->mutex);
            input = getch();
            pthread_mutex_unlock(&data->mutex);
        } else {
            input = ERR;
        }
    } while (!um_tui_is_quit_key(input));
    return 0;
}

/*
 * Update the board. Flags callback function.
 */
static void _um_tui_flags_update_board(Tile **tiles, void *data) {
    CallbackData *cb_data = (CallbackData *) data;
    pthread_mutex_lock(&cb_data->mutex);
    if (cb_data->tiles != tiles) {
        cb_data->tiles = tiles;
    }
    _um_tui_flags_draw_board(tiles, cb_data);
    refresh();
    pthread_mutex_unlock(&cb_data->mutex);
}

/*
 * Update the mine count. Flags callback function.
 */
static void _um_tui_flags_update_mine_count(int mine_count, void *data) {
    CallbackData *cb_data = (CallbackData *) data;
    pthread_mutex_lock(&cb_data->mutex);
    _um_tui_flags_draw_mine_count(mine_count, cb_data);
    refresh();
    pthread_mutex_unlock(&cb_data->mutex);
}

/*
 * Update players. Flags callback function.
 */
static void _um_tui_flags_update_players(const FlagsPlayer *players,
                                            int num_players, void *data) {
    CallbackData *cb_data = (CallbackData *) data;
    pthread_mutex_lock(&cb_data->mutex);
    if (cb_data->players != players) {
        cb_data->players = players;
    }
    _um_tui_flags_draw_players(players, num_players);
    refresh();
    pthread_mutex_unlock(&cb_data->mutex);
}

/*
 * Update the state. Flags callback function.
 */
static void _um_tui_flags_update_state(FlagsState state, void *data) {
    CallbackData *cb_data = (CallbackData *) data;
    pthread_mutex_lock(&cb_data->mutex);
    _um_tui_flags_draw_state(state, cb_data->params.height, cb_data->zoom);
    if (state == DONE) {
        _um_tui_flags_draw_win(cb_data->players,
                        cb_data->params.num_hotseat + cb_data->params.num_ai);
    }
    refresh();
    pthread_mutex_unlock(&cb_data->mutex);
}

/*
 * Update the given tile. Flags callback function.
 */
static void _um_tui_flags_update_tile(const Tile *tile, void *data) {
    CallbackData *cb_data = (CallbackData *) data;
    pthread_mutex_lock(&cb_data->mutex);
    _um_tui_flags_draw_tile(tile, cb_data);
    refresh();
    pthread_mutex_unlock(&cb_data->mutex);
}

/*
 * Update the turn player. Flags callback function.
 */
static void _um_tui_flags_update_turn(int player, void *data) {
    CallbackData *cb_data = (CallbackData *) data;
    pthread_mutex_lock(&cb_data->mutex);
    if (cb_data->turn_player != player) {
        cb_data->turn_player = player;
    }
    _um_tui_flags_draw_turn(player,
                        cb_data->params.num_hotseat + cb_data->params.num_ai);
    refresh();
    pthread_mutex_unlock(&cb_data->mutex);
}

/*
 * Wait for input. Returns 1 if input is ready, 0 if input is not ready, or -1
 * on error.
 */
static int _um_tui_flags_wait(void) {
    #ifdef __MINGW32__
        HANDLE h;
        DWORD result;
        h = GetStdHandle(STD_INPUT_HANDLE);
        if (h == INVALID_HANDLE_VALUE) {
            return -1;
        }
        result = WaitForSingleObject(h, 100);
        if (result == WAIT_TIMEOUT) {
            return 0;
        } else if (result == WAIT_OBJECT_0 &&
                   GetNumberOfConsoleInputEvents(h, &result)) {
            return result > 0;
        }
        return -1;
    #else /* __MINGW32__ */
        fd_set rfds;
        struct timeval timeout = {0, 100000};
        FD_ZERO(&rfds);
        FD_SET(0, &rfds);
        return select(1, &rfds, NULL, NULL, &timeout);
    #endif /* __MINGW32__ */
}

/*
 * Public functions.
 */

/*
 * Flags game.
 */
int um_tui_flags(config_setting_t *config) {
    int result,
        i;
    CallbackData data = {
        PTHREAD_MUTEX_INITIALIZER,
        {0, 0, 0, 0, 0, NULL},
        {0, 0, 2, 0, 0, 1},
        {0, 0},
        NULL,
        NULL,
        0,
        0,
        1
    };
    FlagsAILevel ai_levels[_UM_TUI_FLAGS_MAX_PLAYERS];
    FlagsCallbacks callbacks = {
        _um_tui_flags_update_board,
        _um_tui_flags_update_mine_count,
        _um_tui_flags_update_players,
        _um_tui_flags_update_state,
        _um_tui_flags_update_tile,
        _um_tui_flags_update_turn,
        NULL
    };
    FlagsGame *game;
    for (i = 0; i < _UM_TUI_FLAGS_MAX_PLAYERS; ++i) {
        ai_levels[i] = UM_FLAGS_AI_DEFAULT;
    }
    data.params = UM_FLAGS_DEFAULT;
    data.params.ai_levels = ai_levels;
    _um_tui_flags_config_load(config, &data.options, &data.params, &data.zoom);
    callbacks.data = &data;
    game = um_flags_create(&callbacks, &data.options, &data.params);
    if (!game) {
        return 1;
    }
    result = _um_tui_flags_run(game, &data);
    um_flags_destroy(game);
    pthread_mutex_destroy(&data.mutex);
    _um_tui_flags_config_save(config, &data.options, &data.params, data.zoom);
    return result;
}
